#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.events import subscriber
from zope.component.interfaces import ISite
from zope.interface import implementer
from zope.lifecycleevent.interfaces import IObjectAddedEvent
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.paragraph import IParagraphFactorySettings
from pyams_content.reference.pictograms.interfaces import IPictogramManagerTarget
from pyams_content.shared.common.interfaces import ISharedContentFactory
from pyams_content.shared.common.manager import SharedTool
from pyams_content.shared.form import Form
from pyams_content.shared.form.interfaces import FORM_CONTENT_TYPE, IFormsManager, \
    IFormsManagerFactory
from pyams_utils.adapter import adapter_config
from pyams_utils.registry import utility_config
from pyams_utils.traversing import get_parent


__docformat__ = 'restructuredtext'


@implementer(IFormsManager, IParagraphFactorySettings, IPictogramManagerTarget)
class FormsManager(SharedTool):
    """Forms manager class"""

    shared_content_type = FORM_CONTENT_TYPE
    shared_content_menu = False

    use_captcha = FieldProperty(IFormsManager['use_captcha'])
    default_captcha_client_key = FieldProperty(IFormsManager['default_captcha_client_key'])
    default_captcha_server_key = FieldProperty(IFormsManager['default_captcha_server_key'])
    use_proxy = FieldProperty(IFormsManager['use_proxy'])
    proxy_proto = FieldProperty(IFormsManager['proxy_proto'])
    proxy_host = FieldProperty(IFormsManager['proxy_host'])
    proxy_port = FieldProperty(IFormsManager['proxy_port'])
    proxy_username = FieldProperty(IFormsManager['proxy_username'])
    proxy_password = FieldProperty(IFormsManager['proxy_password'])
    proxy_only_from = FieldProperty(IFormsManager['proxy_only_from'])

    def get_captcha_settings(self):
        result = {
            'use_captcha': False,
            'client_key': None,
            'server_key': None
        }
        if self.use_captcha:
            result.update({
                'use_captcha': True,
                'client_key': self.default_captcha_client_key,
                'server_key': self.default_captcha_server_key
            })
        return result

    def get_proxy_url(self, request):
        if self.use_proxy:
            # check selected domains names
            if self.proxy_only_from:
                domains = map(str.strip, self.proxy_only_from.split(','))
                if request.host not in domains:
                    return None
            return '{}://{}{}:{}/'.format(self.proxy_proto,
                                          '{}{}{}@'.format(self.proxy_username,
                                                           ':' if self.proxy_password else '',
                                                           self.proxy_password or '')
                                          if self.proxy_username else '',
                                          self.proxy_host,
                                          self.proxy_port)


@utility_config(provides=IFormsManagerFactory)
class FormsManagerFactory(object):
    """Defautl forms manager factory"""

    def __new__(cls):
        return FormsManager


@adapter_config(context=IFormsManager, provides=ISharedContentFactory)
def forms_manager_content_factory(context):
    return Form


@subscriber(IObjectAddedEvent, context_selector=IFormsManager)
def handle_added_forms_manager(event):
    """Register forms manager when added"""
    site = get_parent(event.newParent, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, IFormsManager)
