#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from collections import OrderedDict

from persistent import Persistent
from zope.componentvocabulary.vocabulary import UtilityTerm, UtilityVocabulary
from zope.container.contained import Contained
from zope.container.ordered import OrderedContainer
from zope.interface import implementer
from zope.location.interfaces import ISublocations
from zope.schema import Bool, Choice, Date, Int, List, Text, TextLine, URI
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary
from zope.traversing.interfaces import ITraversable

from pyams_content.component.paragraph import BaseParagraph, BaseParagraphFactory, IParagraphFactory
from pyams_content.features.renderer import RenderersVocabulary
from pyams_content.shared.form.interfaces import FORM_FIELDS_PARAGRAPH_NAME, \
    FORM_FIELDS_PARAGRAPH_RENDERERS, FORM_FIELDS_PARAGRAPH_TYPE, FORM_FIELD_CONTAINER_KEY, \
    IFormField, IFormFieldContainer, IFormFieldContainerTarget, IFormFieldFactory, \
    IFormFieldsParagraph, IWfForm
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_global_registry, utility_config
from pyams_utils.request import check_request
from pyams_utils.schema import DottedDecimalField, MailAddressField
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


__docformat__ = 'restructuredtext'

from pyams_content import _


@implementer(IFormField)
class FormField(Persistent, Contained):
    """Form field definition persistent class"""

    name = FieldProperty(IFormField['name'])
    field_type = FieldProperty(IFormField['field_type'])
    label = FieldProperty(IFormField['label'])
    description = FieldProperty(IFormField['description'])
    placeholder = FieldProperty(IFormField['placeholder'])
    values = FieldProperty(IFormField['values'])
    default = FieldProperty(IFormField['default'])
    required = FieldProperty(IFormField['required'])
    visible = FieldProperty(IFormField['visible'])


@adapter_config(context=IFormField, provides=IFormContextPermissionChecker)
def form_field_permission_checker(context):
    """Form field permission checker"""
    form = get_parent(context, IWfForm)
    return IFormContextPermissionChecker(form)


@factory_config(IFormFieldContainer)
class FormFieldContainer(OrderedContainer):
    """Form fields container persistent class"""

    def get_fields(self):
        registry = get_global_registry()
        for field in self.values():
            if field.visible:
                factory = registry.queryUtility(IFormFieldFactory, name=field.field_type)
                if factory is not None:
                    yield factory.get_schema_field(field)

    def find_fields(self, factory):
        for field in self.values():
            if field.visible and (field.field_type == factory):
                yield field


@adapter_config(context=IFormFieldContainerTarget, provides=IFormFieldContainer)
def form_field_container_factory(context):
    """Form fields container factory"""
    return get_annotation_adapter(context, FORM_FIELD_CONTAINER_KEY, IFormFieldContainer,
                                  name='++fields++')


@adapter_config(name='fields', context=IFormFieldContainerTarget, provides=ITraversable)
class FormFieldContainerNamespace(ContextAdapter):
    """Form fields container ++fields++ namespace"""

    def traverse(self, name, firtherpath=None):
        return IFormFieldContainer(self.context)


@adapter_config(name='fields', context=IFormFieldContainerTarget, provides=ISublocations)
class FormFieldsContainerSublocations(ContextAdapter):
    """Form fields container sub-locations adapter"""

    def sublocations(self):
        return IFormFieldContainer(self.context).values()


#
# Form fields factories
#

@vocabulary_config(name='PyAMS form field types')
class FormFieldTypesVocabulary(UtilityVocabulary):
    """Form field types vocabulary"""

    interface = IFormFieldFactory

    def __init__(self, context, **kw):
        request = check_request()
        registry = request.registry
        translate = request.localizer.translate
        utils = [(name, translate(util.label))
                 for (name, util) in sorted(registry.getUtilitiesFor(self.interface),
                                            key=lambda x: x[1].weight)]
        self._terms = OrderedDict((title, UtilityTerm(name, title)) for name, title in utils)

    def __iter__(self):
        return iter(self._terms.values())


class BaseFormFieldFactory(object):
    """Base form field factory"""

    field_factory = None

    def get_schema_field(self, field):
        i18n = II18n(field)
        result = self.field_factory(title=i18n.query_attribute('label'),
                                    description=i18n.query_attribute('description'),
                                    required=field.required,
                                    default=i18n.query_attribute('default'))
        result.__name__ = field.name
        return result


@utility_config(name='textline', provides=IFormFieldFactory)
class TextLineFieldFactory(BaseFormFieldFactory):
    """Textline field factory"""

    label = _("Text")
    weight = 1

    field_factory = TextLine


@utility_config(name='text', provides=IFormFieldFactory)
class TextFieldFactory(BaseFormFieldFactory):
    """Text field factory"""

    label = _("Multi-lines text")
    weight = 2

    field_factory = Text


@utility_config(name='bool', provides=IFormFieldFactory)
class BooleanFieldFactory(BaseFormFieldFactory):
    """Boolean field factory"""

    label = _("Boolean")
    weight = 3

    field_factory = Bool


@utility_config(name='integer', provides=IFormFieldFactory)
class IntegerFieldFactory(BaseFormFieldFactory):
    """Integer field factory"""

    label = _("Integer")
    weight = 4

    field_factory = Int


@utility_config(name='decimal', provides=IFormFieldFactory)
class DecimalFieldFactory(BaseFormFieldFactory):
    """Decimal field factory"""

    label = _("Decimal")
    weight = 5

    field_factory = DottedDecimalField


@utility_config(name='date', provides=IFormFieldFactory)
class DateFieldFactory(BaseFormFieldFactory):
    """Date field factory"""

    label = _("Date")
    weight = 6

    field_factory = Date


@utility_config(name='phone_number', provides=IFormFieldFactory)
class PhoneNumberFieldFactory(BaseFormFieldFactory):
    """Phone number field factory"""

    label = _("Phone number")
    weight = 10

    field_factory = TextLine


@utility_config(name='mail', provides=IFormFieldFactory)
class MailFieldFactory(BaseFormFieldFactory):
    """Mail field factory"""

    label = _("E-mail address")
    weight = 15

    field_factory = MailAddressField


@utility_config(name='uri', provides=IFormFieldFactory)
class URIFieldFactory(BaseFormFieldFactory):
    """URI field factory"""

    label = _("URI")
    weight = 20

    field_factory = URI


class ValuesFieldFactory(BaseFormFieldFactory):
    """Values-based field factory"""


@utility_config(name='choice', provides=IFormFieldFactory)
class ChoiceFieldFactory(ValuesFieldFactory):
    """Choice field factory"""

    label = _("Choice")
    weight = 30

    field_factory = Choice

    def get_schema_field(self, field):
        i18n = II18n(field)
        vocabulary = SimpleVocabulary([
            SimpleTerm(v, title=v)
            for v in field.values
        ])
        result = self.field_factory(title=i18n.query_attribute('label'),
                                    description=i18n.query_attribute('description'),
                                    required=field.required,
                                    default=i18n.query_attribute('default'),
                                    vocabulary=vocabulary)
        result.__name__ = field.name
        return result


@utility_config(name='list', provides=IFormFieldFactory)
class ListFieldFactory(ValuesFieldFactory):
    """List field factory"""

    label = _("List")
    weight = 40

    field_factory = List

    def get_schema_field(self, field):
        i18n = II18n(field)
        vocabulary = SimpleVocabulary([
            SimpleTerm(v, title=v)
            for v in field.values
        ])
        result = self.field_factory(title=i18n.query_attribute('label'),
                                    description=i18n.query_attribute('description'),
                                    required=field.required,
                                    default=[i18n.query_attribute('default')],
                                    value_type=Choice(vocabulary=vocabulary))
        result.__name__ = field.name
        return result


#
# Form fields paragraph
#

@factory_config(provided=IFormFieldsParagraph)
class FormFieldsParagraph(BaseParagraph):
    """Form fields paragraph"""

    factory_name = FORM_FIELDS_PARAGRAPH_TYPE
    icon_class = 'fa-th-list'
    icon_hint = FORM_FIELDS_PARAGRAPH_NAME

    renderer = FieldProperty(IFormFieldsParagraph['renderer'])


@utility_config(name=FORM_FIELDS_PARAGRAPH_TYPE, provides=IParagraphFactory)
class FormFieldsParagraphFactory(BaseParagraphFactory):
    """Form fields paragraph factory"""

    name = FORM_FIELDS_PARAGRAPH_NAME
    content_type = FormFieldsParagraph


@vocabulary_config(name=FORM_FIELDS_PARAGRAPH_RENDERERS)
class FormFieldsRendererVocabulary(RenderersVocabulary):
    """Form fields paragraph renderers vocabulary"""

    content_interface = IFormFieldsParagraph
