#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from zope.interface import alsoProvides, implementer, noLongerProvides, provider
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.paragraph import IParagraphContainerTarget
from pyams_content.features.checker.interfaces import ERROR_VALUE, IContentChecker
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.features.review.interfaces import IReviewTarget
from pyams_content.shared.common import IWfSharedContentFactory, SharedContent, WfSharedContent, WfSharedContentChecker, \
    register_content_type
from pyams_content.shared.form.interfaces import FORM_CONTENT_NAME, FORM_CONTENT_TYPE, IForm, \
    IFormFieldContainer, \
    IFormFieldContainerTarget, IFormHandler, IWfForm, IWfFormFactory, IFormsManager
from pyams_utils.adapter import adapter_config
from pyams_utils.registry import get_global_registry
from pyams_utils.traversing import get_parent


__docformat__ = 'restructuredtext'

from pyams_content import _


@implementer(IWfForm, IFormFieldContainerTarget, IParagraphContainerTarget,
             IPreviewTarget, IReviewTarget)
class WfForm(WfSharedContent):
    """Base form"""

    content_type = FORM_CONTENT_TYPE
    content_name = FORM_CONTENT_NAME

    form_header = FieldProperty(IWfForm['form_header'])
    alt_title = FieldProperty(IWfForm['alt_title'])
    user_title = FieldProperty(IWfForm['user_title'])
    auth_only = FieldProperty(IWfForm['auth_only'])
    submit_label = FieldProperty(IWfForm['submit_label'])
    submit_message = FieldProperty(IWfForm['submit_message'])
    _handler = FieldProperty(IWfForm['handler'])
    override_captcha = FieldProperty(IWfForm['override_captcha'])
    client_captcha_key = FieldProperty(IWfForm['client_captcha_key'])
    server_captcha_key = FieldProperty(IWfForm['server_captcha_key'])
    rgpd_consent = FieldProperty(IWfForm['rgpd_consent'])
    rgpd_warning = FieldProperty(IWfForm['rgpd_warning'])
    rgpd_user_rights = FieldProperty(IWfForm['rgpd_user_rights'])

    def __init__(self):
        super(WfForm, self).__init__()
        self.submit_label = {
            'en': "Submit",
            'fr': "Envoyer"
        }

    @property
    def handler(self):
        return self._handler

    @handler.setter
    def handler(self, value):
        old_handler = self._handler
        if value == old_handler:
            return
        if old_handler is not None:
            handler = self.query_handler(old_handler)
            if (handler is not None) and handler.target_interface:
                noLongerProvides(self, handler.target_interface)
        if value is not None:
            handler = self.query_handler(value)
            if (handler is not None) and handler.target_interface:
                alsoProvides(self, handler.target_interface)
        self._handler = value

    def query_handler(self, handler=None):
        if handler is None:
            handler = self._handler
        if handler:
            registry = get_global_registry()
            return registry.queryUtility(IFormHandler, name=handler)

    def get_captcha_settings(self):
        if self.override_captcha:
            return {
                'use_captcha': bool(self.client_captcha_key),
                'client_key': self.client_captcha_key,
                'server_key': self.server_captcha_key
            }
        else:
            manager = get_parent(self, IFormsManager)
            return manager.get_captcha_settings()


register_content_type(WfForm, shared_content=False)


@provider(IWfFormFactory)
@implementer(IForm)
class Form(SharedContent):
    """Workflow managed form class"""


@adapter_config(context=IWfFormFactory, provides=IWfSharedContentFactory)
def form_content_factory(context):
    return WfForm


@adapter_config(name='properties', context=IWfForm, provides=IContentChecker)
class WfFormContentChecker(WfSharedContentChecker):
    """Form content checker"""

    def inner_check(self, request):
        output = super(WfFormContentChecker, self).inner_check(request)
        if not IFormFieldContainer(self.context).get_fields():
            translate = request.localizer.translate
            output.append(translate(ERROR_VALUE).format(field=translate(_("Form fields")),
                                                        message=translate(_("no field defined"))))
        return output
