#
# Copyright (c) 2015-2022 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

import mimetypes

from pyramid.events import subscriber
from zope.interface import implementer, provider
from zope.lifecycleevent.interfaces import IObjectModifiedEvent
from zope.schema.fieldproperty import FieldProperty

from pyams_content.features.checker.interfaces import IContentChecker, MISSING_VALUE
from pyams_content.shared.common import IWfSharedContentFactory, SharedContent, WfSharedContent, \
    WfSharedContentChecker, register_content_type
from pyams_content.shared.file.interfaces import FILE_CONTENT_NAME, FILE_CONTENT_TYPE, IFile, \
    IWfFile, IWfFileFactory
from pyams_file.interfaces import IFileInfo
from pyams_file.property import FileProperty
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_utils.adapter import adapter_config
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.request import check_request
from pyams_utils.unicode import translate_string


__docformat__ = 'restructuredtext'


from pyams_content import _


@implementer(IWfFile)
class WfFile(WfSharedContent):
    """File persistent class"""

    content_type = FILE_CONTENT_TYPE
    content_name = FILE_CONTENT_NAME

    handle_content_url = True
    handle_header = False
    handle_description = False

    data = FileProperty(IWfFile['data'])
    _filename = FieldProperty(IWfFile['filename'])

    @property
    def filename(self):
        return self._filename

    @filename.setter
    def filename(self, value):
        self._filename = value
        if self.data:
            negotiator = get_utility(INegotiator)
            filename = value.get(negotiator.server_language)
            if filename:
                self.data.filename = translate_string(filename,
                                                      escape_slashes=True,
                                                      force_lower=False)
            elif not self.data.filename:
                title = II18n(self).query_attribute('title')
                extension = mimetypes.guess_extension(self.data.content_type)
                self.data.filename = '{}{}'.format(translate_string(title,
                                                                    spaces='-',
                                                                    force_lower=True),
                                                   extension or '.bin')


register_content_type(WfFile, shared_content=False)


@subscriber(IObjectModifiedEvent, context_selector=IWfFile)
def handle_updated_file(event):
    """Handle updated file"""
    file = event.object
    if not file.data:
        file.filename = {}
        return
    info = IFileInfo(file.data)
    filename = file.filename or {}
    for lang in II18nManager(file).get_languages():
        if not filename.get(lang):
            filename[lang] = info.filename
    file.filename = filename


@provider(IWfFileFactory)
@implementer(IFile)
class File(SharedContent):
    """Workflow managed file persistent class"""


@adapter_config(context=IWfFileFactory,
                provides=IWfSharedContentFactory)
def file_content_factory(context):
    """File content factory"""
    return WfFile


@adapter_config(name='properties',
                context=IWfFile,
                provides=IContentChecker)
class WfFileContentChecker(WfSharedContentChecker):
    """Logo content checker"""

    required_attributes = ('title',)

    def inner_check(self, request):
        output = super().inner_check(request)
        translate = request.localizer.translate
        if not (self.context.data and self.context.data.get_size()):
            output.append(translate(MISSING_VALUE).format(field=translate(IWfFile['data'].title),
                                                          message=translate(_("no content defined"))))
        return output
