#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Any, Contains, Eq, Ge, Le
from pyramid.decorator import reify
from pyramid.view import view_config
from z3c.form import field
from z3c.table.interfaces import IValues
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import implementer
from zope.intid.interfaces import IIntIds
from zope.schema import Choice, Datetime

from pyams_catalog.query import CatalogResultSet
from pyams_content.component.theme.interfaces import ICollectionsManager, \
    ITagsManager, IThemesManager, IThemesManagerTarget
from pyams_content.profile.interfaces import IAdminProfile
from pyams_content.shared.common import CONTENT_TYPES
from pyams_content.shared.common.interfaces import IBaseSharedTool, \
    SHARED_TOOL_WORKFLOW_STATES_VOCABULARY
from pyams_content.shared.common.interfaces.types import DATA_TYPES_VOCABULARY, ITypedSharedTool
from pyams_content.shared.common.interfaces.zmi import ISharedToolDashboardTable
from pyams_content.zmi.interfaces import IAllContentsMenu
from pyams_form.interfaces.form import ISearchFormFactory, ISearchFormResultsFactory
from pyams_form.search import ISearchFields, SearchForm, SearchResultsView, SearchView
from pyams_i18n.interfaces import INegotiator
from pyams_pagelet.interfaces import PageletCreatedEvent
from pyams_pagelet.pagelet import pagelet_config
from pyams_security.schema import Principal
from pyams_sequence.interfaces import ISequentialIntIds
from pyams_sequence.reference import get_last_version
from pyams_skin.interfaces import IContentSearch, IInnerPage, IPageHeader
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.page import DefaultPageHeaderAdapter
from pyams_skin.skin import apply_skin
from pyams_skin.table import BaseTable
from pyams_skin.viewlet.menu import MenuItem
from pyams_template.template import template_config
from pyams_thesaurus.schema import ThesaurusTermsListField
from pyams_thesaurus.zmi.widget import ThesaurusTermsTreeFieldWidget
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.list import unique
from pyams_utils.registry import get_utility
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_utils.vocabulary import vocabulary_config
from pyams_viewlet.viewlet import viewlet_config
from pyams_workflow.interfaces import IWorkflow, IWorkflowVersions
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import AdminView


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Quick search adapters
#

@view_config(name='quick-search.html', context=IBaseSharedTool, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION, renderer='json', xhr=True)
def shared_tool_quick_search_view(request):
    """Shared tool quick search view"""
    results = SharedToolQuickSearchResults(request.context, request)
    if len(results.values) == 1:
        result = results.values[0]
        return {
            'status': 'redirect',
            'location': absolute_url(result, request, 'admin')
        }
    else:
        results.update()
        return {
            'status': 'info',
            'content': {'html': results.render()}
        }


@implementer(ISharedToolDashboardTable)
class SharedToolQuickSearchResults(BaseTable):
    """Shared tool quick search results table"""

    title = _("Quick search results")

    sortOn = None
    dt_sort_order = 'desc'

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes['table'] = {
            'data-ams-datatable-sorting':
                '{0},{1}'.format(len(self.columns) - 1, self.dt_sort_order),
            'data-ams-datatable-display-length':
                IAdminProfile(self.request.principal).table_page_length
        }
        return attributes


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolQuickSearchResults),
                provides=IValues)
class SharedToolQuickSearchValues(ContextRequestViewAdapter):
    """Shared tool quick search results view values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        query = self.request.params.get('query')
        if query:
            sequence = get_utility(ISequentialIntIds)
            query = query.lower().replace('*', '')
            if query.startswith('+'):
                params = Eq(catalog['oid'], sequence.get_full_oid(query))
            else:
                params = Eq(catalog['parents'], intids.register(self.context)) & \
                         Any(catalog['content_type'], CONTENT_TYPES.keys())
                query_params = Eq(catalog['oid'], sequence.get_full_oid(query))
                negotiator = get_utility(INegotiator)
                for lang in {self.request.registry.settings.get('pyramid.default_locale_name',
                                                                'en'),
                             self.request.locale_name,
                             negotiator.server_language} | negotiator.offered_languages:
                    index_name = 'title:{0}'.format(lang)
                    if index_name in catalog:
                        index = catalog[index_name]
                        if index.check_query(query):
                            query_params |= Contains(index,
                                                     ' and '.join((w+'*' for w in query.split())))
                params &= query_params
            return unique(map(get_last_version,
                              CatalogResultSet(CatalogQuery(catalog).query(
                                  params, sort_index='modified_date', reverse=True))))
        else:
            return ()


#
# Advanced search views
#

@vocabulary_config(name=SHARED_TOOL_WORKFLOW_STATES_VOCABULARY)
def WorkflowStatesVocabulary(context):
    """Workflow states vocabulary"""
    target = get_parent(context, IBaseSharedTool)
    if target is not None:
        workflow = IWorkflow(target)
        return workflow.states


@viewlet_config(name='advanced-search.menu', context=IBaseSharedTool, layer=IAdminLayer,
                manager=IAllContentsMenu, permission=VIEW_SYSTEM_PERMISSION, weight=90)
class SharedToolAdvancedSearchMenu(MenuItem):
    """Shared tool advanced search menu"""

    label = _("Advanced search")
    icon_class = None
    url = '#advanced-search.html'


class ISharedToolAdvancedSearchFields(ISearchFields):
    """Shared tool advanced search fields"""

    owner = Principal(title=_("Owner"),
                      required=False)

    status = Choice(title=_("Status"),
                    vocabulary=SHARED_TOOL_WORKFLOW_STATES_VOCABULARY,
                    required=False)

    data_type = Choice(title=_("Data type"),
                       vocabulary=DATA_TYPES_VOCABULARY,
                       required=False)

    created_after = Datetime(title=_("Created after..."),
                             required=False)

    created_before = Datetime(title=_("Created before..."),
                              required=False)

    modified_after = Datetime(title=_("Modified after..."),
                              required=False)

    modified_before = Datetime(title=_("Modified before..."),
                               required=False)

    tags = ThesaurusTermsListField(title=_("Tags"),
                                   required=False)

    themes = ThesaurusTermsListField(title=_("Themes"),
                                     required=False)

    collections = ThesaurusTermsListField(title=_("Collections"),
                                          required=False)


@template_config(template='templates/advanced-search.pt', layer=IPyAMSLayer)
@implementer(IInnerPage)
class SharedToolAdvancedSearchForm(SearchForm):
    """Shared tool advanced search form"""

    legend = _("Advanced search")

    fields_interface = ISharedToolAdvancedSearchFields
    sort_results = True

    prefix = 'search_form.'
    ajax_handler = 'advanced-search-results.html'

    def __init__(self, context, request):
        super().__init__(context, request)
        request.registry.notify(PageletCreatedEvent(self))
        apply_skin(self.request, 'PyAMS admin skin')

    @property
    def fields(self):
        workflow = IWorkflow(self.context)
        fields = field.Fields(self.fields_interface)
        fields['status'].vocabulary = workflow.states
        if not ITypedSharedTool.providedBy(self.context):
            fields = fields.omit('data_type')
        fields['tags'].widgetFactory = ThesaurusTermsTreeFieldWidget
        if IThemesManagerTarget.providedBy(self.context):
            fields['themes'].widgetFactory = ThesaurusTermsTreeFieldWidget
        else:
            fields = fields.omit('themes')
        fields['collections'].widgetFactory = ThesaurusTermsTreeFieldWidget
        return fields

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        if 'tags' in self.widgets:
            widget = self.widgets['tags']
            manager = ITagsManager(self.request.root)
            widget.thesaurus_name = manager.thesaurus_name
            widget.extract_name = manager.extract_name
        if 'themes' in self.widgets:
            widget = self.widgets['themes']
            manager = IThemesManager(self.context)
            widget.thesaurus_name = manager.thesaurus_name
            widget.extract_name = manager.extract_name
        if 'collections' in self.widgets:
            widget = self.widgets['collections']
            manager = ICollectionsManager(self.request.root)
            widget.thesaurus_name = manager.thesaurus_name
            widget.extract_name = manager.extract_name


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolAdvancedSearchForm),
                provides=IContentSearch)
class SharedToolAdvancedSearchFormSearchAdapter(ContextRequestViewAdapter):
    """Shared tool adavanced search form search adapter"""

    def get_params(self, catalog, data):
        intids = get_utility(IIntIds)
        params = Eq(catalog['parents'], intids.register(self.context)) & \
            Any(catalog['content_type'], CONTENT_TYPES.keys())
        query = data.get('query')
        if query:
            sequence = get_utility(ISequentialIntIds)
            if query.startswith('+'):
                params &= Eq(catalog['oid'], sequence.get_full_oid(query))
            else:
                query_params = Eq(catalog['oid'], sequence.get_full_oid(query))
                negotiator = get_utility(INegotiator)
                for lang in {self.request.registry.settings.get('pyramid.default_locale_name',
                                                                'en'),
                             self.request.locale_name,
                             negotiator.server_language} | negotiator.offered_languages:
                    index_name = 'title:{0}'.format(lang)
                    if index_name in catalog:
                        index = catalog[index_name]
                        if index.check_query(query):
                            query_params |= Contains(index,
                                                     ' and '.join((w+'*' for w in query.split())))
                params &= query_params
        if data.get('owner'):
            params &= Eq(catalog['role:owner'], data['owner'])
        if data.get('status'):
            params &= Eq(catalog['workflow_state'], data['status'])
        if data.get('data_type'):
            params &= Eq(catalog['data_type'], data['data_type'])
        if data.get('created_after'):
            params &= Ge(catalog['created_date'], data['created_after'])
        if data.get('created_before'):
            params &= Le(catalog['created_date'], data['created_before'])
        if data.get('modified_after'):
            params &= Ge(catalog['modified_date'], data['modified_after'])
        if data.get('modified_before'):
            params &= Le(catalog['modified_date'], data['modified_before'])
        if data.get('tags'):
            tags = [intids.register(term) for term in data['tags']]
            params &= Any(catalog['tags'], tags)
        if data.get('themes'):
            tags = [intids.register(term) for term in data['themes']]
            params &= Any(catalog['themes'], tags)
        if data.get('collections'):
            tags = [intids.register(term) for term in data['collections']]
            params &= Any(catalog['collections'], tags)
        return params

    def get_search_results(self, data):
        catalog = get_utility(ICatalog)
        params = self.get_params(catalog, data)
        if data.get('status'):
            return unique(map(lambda x: sorted(IWorkflowVersions(x).get_versions(data['status']),
                                               key=lambda y: IZopeDublinCore(y).modified)[0],
                              CatalogResultSet(CatalogQuery(catalog).query(
                                  params, sort_index='modified_date', reverse=True))))
        return unique(map(get_last_version,
                          CatalogResultSet(CatalogQuery(catalog).query(
                              params, sort_index='modified_date', reverse=True))))


@pagelet_config(name='advanced-search.html', context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolAdvancedSearchView(SearchView):
    """Shared tool advanced search view"""

    search_form_factory = SharedToolAdvancedSearchForm


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolAdvancedSearchView),
                provides=IPageHeader)
class SharedToolAdvancedSearchHeaderAdapter(DefaultPageHeaderAdapter):
    """Shared tool advanced search header adapter"""

    back_url = '#dashboard.html'
    back_target = None

    icon_class = 'fa fa-fw fa-search'


@view_config(name='advanced-search-results.html', context=IBaseSharedTool,
             request_type=IPyAMSLayer, permission=VIEW_SYSTEM_PERMISSION)
@implementer(ISharedToolDashboardTable)
class SharedToolAdvancedSearchResultsView(AdminView, SearchResultsView):
    """Shared tool advanced search results view"""

    title = _("Advanced search results")
    search_form_factory = SharedToolAdvancedSearchForm

    sortOn = None
    dt_sort_order = 'desc'

    def __init__(self, context, request):
        super(SharedToolAdvancedSearchResultsView, self).__init__(context, request)
        request.registry.notify(PageletCreatedEvent(self))

    @reify
    def search_form(self):
        form = self.request.registry.queryMultiAdapter((self.context, self.request, self),
                                                       ISearchFormResultsFactory)
        if form is None:
            form = self.request.registry.queryMultiAdapter((self.context, self.request, self),
                                                           ISearchFormFactory)
        if form is None:
            form = self.search_form_factory(self.context, self.request)
        return form

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes['table'] = {
            'data-ams-datatable-display-length':
                IAdminProfile(self.request.principal).table_page_length
        }
        if self.search_form.sort_results:
            attributes['table']['data-ams-datatable-sorting'] = \
                "{0},{1}".format(len(self.columns) - 1, self.dt_sort_order)
        else:
            attributes['table']['data-ams-datatable-sorting'] = '[]'
        return attributes


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolAdvancedSearchResultsView),
                provides=IValues)
class SearchResultsViewValuesAdapter(ContextRequestViewAdapter):
    """Search results view values adapter"""

    @property
    def values(self):
        return self.view.search_form.get_search_results() or ()
