#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Any, Eq, Or
from z3c.table.interfaces import IColumn, IValues
from zope.interface import Interface, implementer

from pyams_catalog.query import CatalogResultSet
from pyams_content.shared.common.interfaces import IWfSharedContent
from pyams_content.shared.common.interfaces.zmi import ISiteRootDashboardTable
from pyams_content.shared.site.interfaces import ISiteContainer
from pyams_pagelet.pagelet import pagelet_config
from pyams_portal.interfaces import IPortalTemplate
from pyams_sequence.interfaces import ISequentialIdInfo
from pyams_skin.container import ContainerView
from pyams_skin.interfaces import IInnerPage
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import BaseTable, NameColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces import ICacheKeyValue, VIEW_SYSTEM_PERMISSION
from pyams_utils.list import unique_iter_max
from pyams_utils.registry import get_utility
from pyams_utils.traversing import get_parent
from pyams_viewlet.viewlet import viewlet_config
from pyams_workflow.interfaces import IWorkflowState
from pyams_zmi.interfaces.menu import IContentManagementMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import AdminView


__docformat__ = 'restructuredtext'

from pyams_content import _


@viewlet_config(name='reverse-links.menu', context=IWfSharedContent, layer=IAdminLayer,
                manager=IContentManagementMenu, permission=VIEW_SYSTEM_PERMISSION, weight=40)
class SequentialITargetReverseLinksMenu(MenuItem):
    """Sequential ID target reverse links menu"""

    label = _("Reverse links")
    icon_class = 'fa-anchor'
    url = '#reverse-links.html'


@implementer(ISiteRootDashboardTable)
class SequentialIdTargetReverseLinkTable(BaseTable):
    """Sequential ID target reverse links table"""

    title = _("Content's internal links")


@adapter_config(name='name', context=(Interface, IPyAMSLayer, SequentialIdTargetReverseLinkTable), provides=IColumn)
class ReverseLinkNameColumn(NameColumn):
    """Reverse link name column"""

    _header = _("Title")


@adapter_config(context=(IWfSharedContent, IPyAMSLayer, SequentialIdTargetReverseLinkTable), provides=IValues)
class SequentialIdTargetReverseLinkValues(ContextRequestViewAdapter):
    """Sequential ID target reverse links values"""

    @property
    def values(self):

        def get_item(result):
            parent = get_parent(result, IWfSharedContent)
            if parent is not None:
                return parent
            parent = get_parent(result, IPortalTemplate)
            if parent is None:
                parent = get_parent(result, ISiteContainer)
            if parent is None:
                parent = self.request.root
            return parent

        catalog = get_utility(ICatalog)
        oid = ISequentialIdInfo(self.context).hex_oid
        params = Or(Eq(catalog['link_reference'], oid),
                    Any(catalog['link_references'], {oid}))
        return unique_iter_max(map(get_item,
                                   CatalogResultSet(CatalogQuery(catalog).query(params))),
                               key=lambda x: getattr(ISequentialIdInfo(x, None),
                                                     'oid', ICacheKeyValue(x)),
                               sort_key=lambda x: getattr(IWorkflowState(x, None),
                                                          'version_id', ICacheKeyValue(x)))


@pagelet_config(name='reverse-links.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@implementer(IInnerPage)
class SequentialIdTargetReverseLinkView(AdminView, ContainerView):
    """Sequential ID target reverse links view"""

    table_class = SequentialIdTargetReverseLinkTable
