#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.events import subscriber
from z3c.form import field
from z3c.form.interfaces import HIDDEN_MODE, IDataExtractedEvent
from zope.interface import implementer

from pyams_content.interfaces import IBaseContent, MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.interfaces import IWfSharedContent
from pyams_content.shared.common.zmi import WfSharedContentHeaderAdapter
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IWidgetForm
from pyams_i18n.widget import I18nSEOTextLineFieldWidget
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_widget_refresh_event
from pyams_skin.interfaces import IInnerPage, IPageHeader
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.url import generate_url
from pyams_viewlet.manager import viewletmanager_config
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminEditForm
from pyams_zmi.interfaces import IPropertiesEditForm
from pyams_zmi.interfaces.menu import IContentManagementMenu, IPropertiesMenu
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Content properties
#

@viewlet_config(name='content.menu',
                context=IWfSharedContent, layer=IAdminLayer,
                manager=IContentManagementMenu, weight=10,
                permission=VIEW_SYSTEM_PERMISSION)
@viewletmanager_config(name='properties.menu',
                       layer=IAdminLayer, provides=IPropertiesMenu)
@implementer(IPropertiesMenu)
class SharedContentCompositionMenu(MenuItem):
    """Shared content composition menu"""

    label = _("Composition")
    icon_class = 'fa-dropbox'
    url = '#'


@viewlet_config(name='properties.menu',
                context=IWfSharedContent, layer=IAdminLayer,
                manager=IPropertiesMenu, weight=10,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedContentPropertiesMenu(MenuItem):
    """Shared content properties menu"""

    label = _("Properties")
    icon_class = 'fa-edit'
    url = '#properties.html'


@pagelet_config(name='properties.html', context=IWfSharedContent, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json', context=IWfSharedContent, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION)
@implementer(IPropertiesEditForm, IWidgetForm, IInnerPage)
class SharedContentPropertiesEditForm(AdminEditForm):
    """Shared content properties edit form"""

    legend = _("Content properties")

    interface = IWfSharedContent
    fieldnames = ('title', 'short_name', 'content_url', 'header', 'description', 'notepad')

    @property
    def fields(self):
        fields = field.Fields(self.interface).select(*self.fieldnames)
        if 'title' in fields:
            fields['title'].widgetFactory = I18nSEOTextLineFieldWidget
        if ('content_url' in fields) and not self.context.handle_content_url:
            fields = fields.omit('content_url')
        if ('header' in fields) and not self.context.handle_header:
            fields = fields.omit('header')
        if ('description' in fields) and not self.context.handle_description:
            fields = fields.omit('description')
        return fields

    def updateWidgets(self, prefix=None):
        super(SharedContentPropertiesEditForm, self).updateWidgets(prefix)
        if 'short_name' in self.widgets:
            self.widgets['short_name'].set_mode(HIDDEN_MODE)

    def get_ajax_output(self, changes):
        updated = changes.get(IBaseContent, ())
        if 'title' in updated:
            return {
                'status': 'redirect',
                'message': self.request.localizer.translate(self.successMessage)
            }
        else:
            output = super(self.__class__, self).get_ajax_output(changes)
            updated = changes.get(IWfSharedContent, ())
            if 'content_url' in updated:
                output.setdefault('events', []).append(
                    get_json_widget_refresh_event(self.context, self.request,
                                                  SharedContentPropertiesEditForm, 'content_url'))
            return output


@subscriber(IDataExtractedEvent, form_selector=SharedContentPropertiesEditForm)
def handle_content_properties_data_extraction(event):
    """Automatically set short_name as title"""
    data = event.data
    if not event.form.context.handle_short_name:
        data['short_name'] = data['title'].copy()
    if 'content_url' in data:
        data['content_url'] = generate_url(data['content_url'])


@adapter_config(context=(IWfSharedContent, IAdminLayer, SharedContentPropertiesEditForm),
                provides=IPageHeader)
class SharedContentPropertiesHeaderAdapter(WfSharedContentHeaderAdapter):
    """Shared content properties header adapter"""

    icon_class = 'fa fa-fw fa-edit'
