#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json
from datetime import datetime

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Any, Eq, NotAny
from pyramid.decorator import reify
from pyramid.response import Response
from pyramid.view import view_config
from z3c.form import button, field
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
from z3c.form.interfaces import HIDDEN_MODE
from z3c.table.column import CheckBoxColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import Interface, alsoProvides, implementer
from zope.intid.interfaces import IIntIds
from zope.lifecycleevent import ObjectModifiedEvent
from zope.schema import Bool, Choice, TextLine
from zope.schema.vocabulary import getVocabularyRegistry

from pyams_catalog.query import CatalogResultSet
from pyams_content.interfaces import MANAGE_SITE_PERMISSION, MANAGE_TOOL_PERMISSION
from pyams_content.profile.interfaces import IAdminProfile
from pyams_content.shared.common import CONTENT_TYPES
from pyams_content.shared.common.interfaces import IBaseSharedTool, ISharedTool, ISharedToolRoles, \
    IWfSharedContent, IWfSharedContentRoles, SHARED_TOOL_CONTRIBUTORS_VOCABULARY
from pyams_content.shared.common.interfaces.zmi import ISharedToolDashboardTable
from pyams_form.form import AJAXAddForm, AddForm, ajax_config
from pyams_form.help import FormHelp
from pyams_form.interfaces.form import IFormHelp, IInnerForm, ISearchFormFactory, \
    ISearchFormResultsFactory
from pyams_form.schema import CloseButton
from pyams_form.search import SearchForm, SearchResultsView, SearchView
from pyams_form.widget import HiddenSelect2FieldWidget
from pyams_pagelet.interfaces import PageletCreatedEvent
from pyams_pagelet.pagelet import pagelet_config
from pyams_security.principal import MissingPrincipal
from pyams_security.schema import Principal
from pyams_security.utility import get_principal
from pyams_security.zmi.interfaces import IObjectSecurityMenu
from pyams_sequence.interfaces import ISequentialIdInfo, ISequentialIntIds
from pyams_sequence.reference import get_last_version
from pyams_skin.interfaces import IContentSearch
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.skin import apply_skin
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.list import unique
from pyams_utils.registry import get_utility
from pyams_viewlet.viewlet import viewlet_config
from pyams_workflow.interfaces import IWorkflow, IWorkflowState, IWorkflowVersions
from pyams_workflow.versions import WorkflowHistoryItem
from pyams_zmi.form import AdminDialogAddForm
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import AdminView


__docformat__ = 'restructuredtext'

from pyams_content import _


@viewlet_config(name='change-owner.menu',
                context=IWfSharedContent, layer=IAdminLayer,
                view=Interface, manager=IObjectSecurityMenu, weight=10,
                permission=MANAGE_SITE_PERMISSION)
class WfSharedContentOwnerChangeMenu(MenuItem):
    """Shared content owner change menu"""

    label = _("Change owner...")
    icon_class = 'fa fa-fw fa-user'

    url = 'change-owner.html'
    modal_target = True


class IWfSharedContentOwnerChangeInfo(Interface):
    """Shared content owner change form fields"""

    new_owner = Principal(title=_("New owner"),
                          description=_("The selected user will become the new content's owner"))

    keep_owner_as_contributor = Bool(title=_("Keep previous owner as contributor"),
                                     description=_("If 'yes', the previous owner will still be "
                                                   "able to modify this content"),
                                     required=False,
                                     default=False)


class IWfSharedContentOwnerChangeButtons(Interface):
    """Shared content owner change form buttons"""

    close = CloseButton(name='close', title=_("Cancel"))
    change = button.Button(name='change', title=_("Change owner"))


@pagelet_config(name='change-owner.html',
                context=IWfSharedContent, layer=IPyAMSLayer,
                permission=MANAGE_SITE_PERMISSION)
@ajax_config(name='change-owner.json',
             context=IWfSharedContent, layer=IPyAMSLayer,
             base=AJAXAddForm)
class WfSharedContentOwnerChangeForm(AdminDialogAddForm):
    """Shared content owner change form"""

    legend = _("Change content's owner")

    fields = field.Fields(IWfSharedContentOwnerChangeInfo)
    fields['keep_owner_as_contributor'].widgetFactory = SingleCheckBoxFieldWidget
    buttons = button.Buttons(IWfSharedContentOwnerChangeButtons)

    edit_permission = MANAGE_SITE_PERMISSION

    def updateActions(self):
        super(WfSharedContentOwnerChangeForm, self).updateActions()
        if 'change' in self.actions:
            self.actions['change'].addClass('btn-primary')

    def createAndAdd(self, data):
        data = data.get(self, data)
        new_owner = data.get('new_owner')
        workflow = IWorkflow(self.context)
        translate = self.request.localizer.translate
        for version in IWorkflowVersions(self.context).get_versions():
            state = IWorkflowState(version)
            if state.state in workflow.readonly_states:
                continue
            roles = IWfSharedContentRoles(version)
            [previous_owner] = roles.owner
            roles.owner = {new_owner}
            contributors = roles.contributors.copy()  # don't modify contributors in place!!
            if data.get('keep_owner_as_contributor'):
                if previous_owner not in contributors:
                    contributors.add(previous_owner)
            else:
                if previous_owner in contributors:
                    contributors.remove(previous_owner)
            roles.contributors = contributors
            state.history.append(
                WorkflowHistoryItem(date=datetime.utcnow(),
                                    source_state=state.state,
                                    target_state=state.state,
                                    transition_id='--',
                                    principal=self.request.principal.id,
                                    comment=translate(_("Owner changed: {} -> {}")).format(
                                        get_principal(self.request, previous_owner).title,
                                        get_principal(self.request, new_owner).title
                                    )))
            self.request.registry.notify(ObjectModifiedEvent(version))

    def get_ajax_output(self, changes):
        return {'status': 'reload'}


@adapter_config(context=(IWfSharedContent, IAdminLayer, WfSharedContentOwnerChangeForm),
                provides=IFormHelp)
class WfSharedContentOwnerChangeFormHelp(FormHelp):
    """Shared content owner change form help"""

    message = _("All versions of this content which are not archived will be transferred to "
                "newly selected owner")
    message_format = 'rest'


#
# Bulk owner change form
#

@viewlet_config(name='change-owner.menu',
                context=ISharedTool, layer=IAdminLayer,
                view=Interface, manager=IObjectSecurityMenu, weight=930,
                permission=MANAGE_TOOL_PERMISSION)
class SharedToolOwnerChangeMenu(MenuItem):
    """Shared tool owner change menu"""

    label = _("Change owner...")
    icon_class = 'fa fa-fw fa-user'

    url = '#change-owner.html'
    modal_target = False


class ISharedToolOwnerSearchFormFields(Interface):
    """Shared tool owner change form fields interface"""

    old_owner = Choice(title=_("Current owner"),
                       description=_("Select current contributor for which contents should be "
                                     "affected to a new owner"),
                       vocabulary=SHARED_TOOL_CONTRIBUTORS_VOCABULARY,
                       required=True)


class SharedToolOwnerSearchForm(SearchForm):
    """Shared tool owner search form"""

    legend = _("Search owner contents")

    fields = field.Fields(ISharedToolOwnerSearchFormFields)
    fields['old_owner'].widgetFactory = HiddenSelect2FieldWidget

    sort_results = True

    prefix = 'owner_form.'
    ajax_handler = 'owner-search-results.html'

    def __init__(self, context, request):
        super().__init__(context, request)
        request.registry.notify(PageletCreatedEvent(self))
        apply_skin(self.request, 'PyAMS admin skin')

    def get_owners(self):
        """Owners list getter"""
        vr = getVocabularyRegistry()
        vocabulary = vr.get(self.context, SHARED_TOOL_CONTRIBUTORS_VOCABULARY)
        results = []
        for term in vocabulary:
            results.append({
                'id': term.value,
                'text': term.title
            })
        return results

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        owner = self.widgets.get('old_owner')
        if owner is not None:
            owner.prompt = True
            owner.promptMessage = _("Please select current owner...")
            owner.object_data = {
                'ams-select2-enable-free-tags': True,
                'ams-select2-data': json.dumps(self.get_owners()),
                'ams-change-handler': 'MyAMS.form.autoSubmit'
            }
            alsoProvides(owner, IObjectData)


@adapter_config(context=(IBaseSharedTool, IAdminLayer, SharedToolOwnerSearchForm),
                provides=IContentSearch)
class SharedToolOwnerSearchFormSearchAdapter(ContextRequestViewAdapter):
    """Shared tool owner search form search results adapter"""

    def get_search_results(self, data):
        owner = self.request.params.get('{}{}old_owner'.format(self.view.prefix,
                                                               self.view.widgets.prefix))
        if not owner:
            return []
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        workflow = IWorkflow(self.context)
        params = Eq(catalog['parents'], intids.register(self.context)) & \
            Any(catalog['content_type'], CONTENT_TYPES.keys()) & \
            NotAny(catalog['workflow_state'], workflow.archived_states) & \
            Eq(catalog['role:owner'], owner)
        return unique(map(get_last_version,
                          CatalogResultSet(CatalogQuery(catalog).query(
                              params, sort_index='modified_date', reverse=True))))


@pagelet_config(name='change-owner.html',
                context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
class SharedToolOwnerSearchView(SearchView):
    """Shared tool owner search view"""

    search_form_factory = SharedToolOwnerSearchForm


#
# Owner change form
#

class IOwnerChangeFormFields(Interface):
    """Owner change form fields interface"""

    selection = TextLine(title=_("Selected items"),
                         required=False)

    new_owner = Principal(title=_("New owner"),
                          description=_("Selected contents ownership will be assigned to "
                                        "selected principal"),
                          required=True)

    set_as_tool_contributor = Bool(title=_("Set as tool contributor"),
                                   description=_("If 'yes', selected owner will be granted the "
                                                 "contributor role on shared tool; otherwise, he "
                                                 "will get this role only on selected contents"),
                                   required=False,
                                   default=False)

    keep_owner_as_contributor = Bool(title=_("Keep previous owner as contributor"),
                                     description=_("If 'yes', the previous owner will still be "
                                                   "able to modify selected contents"),
                                     required=False,
                                     default=False)


class IOwnerChangeFormActions(Interface):
    """Owner change form actions interface"""

    assign = button.Button(name='assign', title=_("Change owner"))


@ajax_config(name='change-owner.json',
             context=IBaseSharedTool, layer=IPyAMSLayer,
             permission=MANAGE_TOOL_PERMISSION,
             base=AJAXAddForm)
@implementer(IInnerForm, IObjectData)
class SharedToolOwnerChangeForm(AddForm):
    """Shared tool owner change form"""

    legend = _("New contents owner")

    prefix = 'owner_form.'
    fields = field.Fields(IOwnerChangeFormFields)
    fields['set_as_tool_contributor'].widgetFactory = SingleCheckBoxFieldWidget
    fields['keep_owner_as_contributor'].widgetFactory = SingleCheckBoxFieldWidget

    buttons = button.Buttons(IOwnerChangeFormActions)
    handleActions = True

    @property
    def object_data(self):
        return {
            'ams-form-data-init-callback': "MyAMS.container.getCheckedElements",
            'ams-container-source': 'owner_form.widgets.selection-input',
            'ams-container-target': 'owner_form.widgets.selection'
        }

    def updateWidgets(self, prefix=None):
        super().updateWidgets()
        selection = self.widgets.get('selection')
        if selection is not None:
            selection.mode = HIDDEN_MODE

    def updateActions(self):
        super().updateActions()
        assign = self.actions.get('assign')
        if assign is not None:
            assign.addClass('btn-primary')

    @button.handler(buttons['assign'])
    def handleAssign(self, action):
        data, errors = self.extractData()
        if errors:
            self.status = self.formErrorsMessage
            return
        selection = data.get('selection')
        if not selection:
            self.status = self.request.localizer.translate(_("Can't change owner for empty "
                                                             "selection!"))
            return
        registry = self.request.registry
        translate = self.request.localizer.translate
        selection = list(map(int, selection.split(',')))
        new_owner = data.get('new_owner')
        new_principal = get_principal(self.request, new_owner)
        set_as_tool_contributor = data.get('set_as_tool_contributor')
        if set_as_tool_contributor:
            tool_roles = ISharedToolRoles(self.context)
            tool_contributors = tool_roles.contributors.copy()  # don't update roles in place!!!
            if new_owner not in tool_contributors:
                tool_contributors |= {new_owner}
                tool_roles.contributors = tool_contributors
        sequential_ids = get_utility(ISequentialIntIds)
        for oid in selection:
            content = sequential_ids.queryObject(oid)
            if content is not None:
                workflow = IWorkflow(content)
                for version in IWorkflowVersions(content).get_versions():
                    state = IWorkflowState(version)
                    if state.state in workflow.archived_states:
                        continue
                    roles = IWfSharedContentRoles(version)
                    contributors = roles.contributors.copy()  # don't update roles in place!!!
                    [old_owner] = roles.owner
                    roles.owner = {new_owner}
                    old_principal = get_principal(self.request, principal_id=old_owner)
                    if data.get('keep_owner_as_contributor'):
                        if (old_principal is not None) and \
                                not isinstance(old_principal, MissingPrincipal):
                            contributors |= {old_owner}
                    else:
                        if old_owner in contributors:
                            contributors.remove(old_owner)
                    roles.contributors = contributors
                    state.history.append(
                        WorkflowHistoryItem(date=datetime.utcnow(),
                                            source_state=state.state,
                                            target_state=state.state,
                                            transition_id='--',
                                            principal=self.request.principal.id,
                                            comment=translate(
                                                _("Owner changed: {} -> {}")).format(
                                                old_principal.title,
                                                new_principal.title
                                            )))
                    registry.notify(ObjectModifiedEvent(version))
        return selection

    def get_ajax_output(self, changes):
        translate = self.request.localizer.translate
        if not changes:
            return {
                'status': 'info',
                'message': translate(_("No owner changed for empty selection"))
            }
        return {
            'status': 'success',
            'message': translate(_("Owner changed successfully for {} contents")).format(
                len(changes)),
            'callbacks': [{
                'callback': "MyAMS.container.removeCheckedElements",
                'options': {
                    'source': 'owner_form.widgets.selection-input'
                }
            }]
        }


#
# Owner search results view
#

@view_config(name='owner-search-results.html',
             context=IBaseSharedTool, request_type=IPyAMSLayer,
             permission=MANAGE_TOOL_PERMISSION)
@implementer(ISharedToolDashboardTable)
class SharedToolOwnerSearchResultsView(AdminView, SearchResultsView):
    """Shared tool owner search results view"""

    title = _("Owner search results")
    search_form_factory = SharedToolOwnerSearchForm
    change_owner_form = None

    sortOn = None
    dt_sort_order = 'desc'

    def __init__(self, context, request):
        super().__init__(context, request)
        request.registry.notify(PageletCreatedEvent(self))

    @reify
    def search_form(self):
        form = self.request.registry.queryMultiAdapter((self.context, self.request, self),
                                                       ISearchFormResultsFactory)
        if form is None:
            form = self.request.registry.queryMultiAdapter((self.context, self.request, self),
                                                           ISearchFormFactory)
        if form is None:
            form = self.search_form_factory(self.context, self.request)
        return form

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes['table'] = {
            'data-ams-datatable-display-length':
                IAdminProfile(self.request.principal).table_page_length
        }
        if self.search_form.sort_results:
            attributes['table']['data-ams-datatable-sorting'] = \
                "{0},{1}".format(len(self.columns) - 1, self.dt_sort_order)
        else:
            attributes['table']['data-ams-datatable-sorting'] = '[]'
        return attributes

    def update(self):
        super().update()
        if len(self.values) > 0:
            self.change_owner_form = SharedToolOwnerChangeForm(self.context, self.request)
            self.change_owner_form.update()

    def __call__(self):
        self.update()
        result = self.renderTable()
        if self.change_owner_form:
            result += self.change_owner_form.render()
        return Response(result)


@adapter_config(name='checked',
                context=(IBaseSharedTool, IAdminLayer, SharedToolOwnerSearchResultsView),
                provides=IColumn)
class SharedToolOwnerSearchResultsCheckColumn(CheckBoxColumn):
    """Shared tool owner search results check column"""

    dt_sortable = 'false'
    weight = 1

    def renderHeadCell(self):
        return '<input type="checkbox" ' \
               'data-ams-change-handler="MyAMS.container.selectAllElements" />'

    def getItemKey(self, item):
        return 'owner_form.widgets.selection-input'

    def getItemValue(self, item):
        return str(ISequentialIdInfo(item).oid)

    def isSelected(self, item):
        return False

    def renderCell(self, item):
        return super().renderCell(item).replace(' />',
                                                ' data-ams-click-handler="MyAMS.container.checkElement" '
                                                ' data-ams-stop-propagation="true" />')


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolOwnerSearchResultsView),
                provides=IValues)
class SharedToolOwnerSearchResultsViewValuesAdapter(ContextRequestViewAdapter):
    """Search results view values adapter"""

    @property
    def values(self):
        return self.view.search_form.get_search_results() or ()
