#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from z3c.form import field
from z3c.form.interfaces import DISPLAY_MODE, INPUT_MODE
from zope.interface import Interface, implementer

from pyams_content.interfaces import MANAGE_TOOL_PERMISSION
from pyams_content.shared.common.interfaces import IBaseSharedTool
from pyams_form.form import ajax_config
from pyams_form.help import FormHelp
from pyams_form.interfaces.form import IFormHelp, IWidgetForm
from pyams_i18n.interfaces import II18n, II18nManager
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces import IContentTitle, IInnerPage, IPageHeader
from pyams_skin.interfaces.viewlet import IBreadcrumbItem, IMenuHeader
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.page import DefaultPageHeaderAdapter
from pyams_skin.viewlet.breadcrumb import BreadcrumbAdminLayerItem
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.adapter import ContextRequestAdapter, ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces import MANAGE_SYSTEM_PERMISSION
from pyams_viewlet.manager import viewletmanager_config
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminEditForm
from pyams_zmi.interfaces import IPropertiesEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu, ISiteManagementMenu
from pyams_zmi.layer import IAdminLayer

__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Shared tools common adapters
#

@adapter_config(context=(IBaseSharedTool, IAdminLayer), provides=IBreadcrumbItem)
class SharedToolBreadcrumbAdapter(BreadcrumbAdminLayerItem):
    """Shared tool breadcrumb adapter"""

    @property
    def label(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    css_class = 'strong'


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, Interface), provides=IContentTitle)
class SharedToolTitleAdapter(ContextRequestViewAdapter):
    """Shared tool title adapter"""

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)


@adapter_config(context=(IBaseSharedTool, ISiteManagementMenu), provides=IMenuHeader)
class SharedToolSiteManagementMenuHeader(ContextRequestAdapter):
    """Shared tool site management menu header adapter"""

    @property
    def header(self):
        if self.context.shared_content_menu:
            return _("Content management")
        else:
            return _("Tool management")


#
# Shared tool properties
#

@viewlet_config(name='properties.menu', context=IBaseSharedTool, layer=IAdminLayer,
                manager=ISiteManagementMenu, permission=MANAGE_TOOL_PERMISSION, weight=15)
@viewletmanager_config(name='properties.menu', layer=IAdminLayer, provides=IPropertiesMenu)
@implementer(IPropertiesMenu)
class SharedToolPropertiesMenu(MenuItem):
    """Shared tool properties menu"""

    label = _("Properties")
    icon_class = 'fa-edit'
    url = '#properties.html'


@pagelet_config(name='properties.html', context=IBaseSharedTool, layer=IPyAMSLayer, permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='properties.json', context=IBaseSharedTool, layer=IPyAMSLayer)
@implementer(IWidgetForm, IInnerPage, IPropertiesEditForm)
class SharedToolPropertiesEditForm(AdminEditForm):
    """Shared tool properties edit form"""

    legend = _("Shared tool properties")

    fields = field.Fields(IBaseSharedTool).omit('__parent__', '__name__')

    edit_permission = MANAGE_TOOL_PERMISSION

    def updateWidgets(self, prefix=None):
        super(SharedToolPropertiesEditForm, self).updateWidgets(prefix)
        if not self.request.has_permission(MANAGE_SYSTEM_PERMISSION, context=self.context):
            self.widgets['shared_content_workflow'].mode = DISPLAY_MODE
        if self.mode == INPUT_MODE:
            translate = self.request.localizer.translate
            for name in ('label', 'facets_label'):
                label = self.widgets.get(name)
                if label is not None:
                    label.after_widget_notice = \
                        '<div class="alert-warning padding-5">{}</div>'.format(
                            translate(_("WARNING: if contents already exist in this content type, "
                                        "updating this label will require a complete index rebuild "
                                        "of Elasticsearch index for this content type")))


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolPropertiesEditForm), provides=IFormHelp)
class SharedToolPropertiesHelpAdapter(FormHelp):
    """Shared tool properties help adapter"""

    permission = MANAGE_SYSTEM_PERMISSION

    header = _("WARNING")
    status = 'danger'
    message = _("""Workflow shouldn't be modified if this tool already contains any shared content!""")
    message_format = 'rest'


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, Interface), provides=IPageHeader)
class SharedToolPropertiesHeaderAdapter(DefaultPageHeaderAdapter):
    """Shared tool properties header adapter"""

    back_url = '/admin#properties.html'
    back_target = None

    icon_class = 'fa fa-fw fa-edit'


#
# Shared tool languages
#

@viewlet_config(name='languages.menu', context=IBaseSharedTool, layer=IAdminLayer,
                manager=ISiteManagementMenu, permission=MANAGE_TOOL_PERMISSION, weight=100)
class SharedToolLanguagesMenu(MenuItem):
    """Shared tool languages menu"""

    label = _("Languages")
    icon_class = 'fa-flag'

    url = '#languages.html'


@pagelet_config(name='languages.html', context=IBaseSharedTool, layer=IPyAMSLayer, permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='languages.json', context=IBaseSharedTool, layer=IPyAMSLayer)
@implementer(IInnerPage, IWidgetForm)
class SharedToolLanguagesEditForm(AdminEditForm):
    """Shared tool languages edit form"""

    legend = _("Content languages")

    fields = field.Fields(II18nManager)
    edit_permission = MANAGE_TOOL_PERMISSION


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolLanguagesEditForm), provides=IFormHelp)
class SharedToolLanguagesEditFormHelp(FormHelp):
    """Shared tool languages edit form help"""

    message = _("Tool languages are used to translate own tool properties, and newly created contents will propose "
                "these languages by default")
    message_format = 'rest'
