#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from datetime import datetime
from pyramid.decorator import reify
from zope.interface import Interface

from pyams_content.profile.interfaces import IAdminProfile
from pyams_content.shared.common.interfaces import IWfSharedContent, IBaseSharedTool
from pyams_form.interfaces.form import IInnerTabForm
from pyams_i18n.interfaces import II18n
from pyams_security.interfaces import ISecurityManager
from pyams_sequence.interfaces import ISequentialIntIds, ISequentialIdInfo
from pyams_skin.interfaces import IContextTitlePrefix
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.site import ContextTitlePrefixAdapter
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.date import format_datetime
from pyams_utils.registry import get_utility
from pyams_utils.timezone import tztime
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import contentprovider_config
from pyams_workflow.interfaces import IWorkflowState, IWorkflow, IWorkflowStateLabel, IWorkflowVersions, \
    IWorkflowPublicationInfo

from pyams_content import _


@adapter_config(context=(IWfSharedContent, IPyAMSLayer), provides=IContextTitlePrefix)
class SharedContentTitlePrefixAdapter(ContextTitlePrefixAdapter):
    """Shared content title prefix adapter"""

    @property
    def prefix(self):
        result = super(SharedContentTitlePrefixAdapter, self).prefix
        title = II18n(self.context).query_attribute('short_name', request=self.request)
        if result and title:
            return '{result} > {title}'.format(**locals())
        else:
            return result or title


@contentprovider_config(name='content_header', context=IWfSharedContent, view=Interface, layer=IPyAMSLayer)
@template_config(template='templates/header.pt', layer=IPyAMSLayer)
class SharedContentHeaderContentProvider(object):
    """Header for shared contents"""

    back_url = '#properties.html'
    back_target = None

    icon_class = ''

    def __init__(self, context, request, view):
        super(SharedContentHeaderContentProvider, self).__init__(context, request, view)
        # sequence
        sequence = get_utility(ISequentialIntIds)
        self.oid = sequence.get_short_oid(ISequentialIdInfo(context).oid)
        # security
        security = get_utility(ISecurityManager)
        owner = next(iter(context.owner))
        self.owner = security.get_principal(owner).title
        # workflow
        translate = request.localizer.translate
        workflow = IWorkflow(context)
        versions = IWorkflowVersions(context)
        state = IWorkflowState(context)
        self.version_id = state.version_id
        # init state format
        state_format = translate(_("{state} by {principal}"))
        if state.state_urgency:
            state_format = state_format.replace('{state}',
                                                '{state} '
                                                '<i class="fa fa-fw fa-exclamation-triangle txt-color-red"></i>')
        elif state.state in workflow.published_states:
            pub_info = IWorkflowPublicationInfo(context, None)
            if (pub_info is not None) and not pub_info.is_published():
                now = tztime(datetime.utcnow())
                if pub_info.publication_expiration_date and (pub_info.publication_expiration_date < now):
                    icon = 'fa-warning'
                    message = translate(_("Content publication expiration date is outdated"))
                else:
                    icon = 'fa-hourglass-half'
                    message = translate(_("Content publication start date is not passed yet"))
                state_format = state_format.replace('{state}',
                                                    '{{state}} <i class="fa fa-fw {} font-xs '
                                                    'text-danger opacity-75 hint align-base" '
                                                    'data-ams-hint-offset="5" title="{}"></i>'.format(icon, message))
        state_class = 'text-danger'
        state_format = state_format.replace('{state}',
                                            '<span class="{0}">{{state}}</span>'.format(state_class))
        # init principal format
        principal_class = 'text-danger' \
            if state.state in (workflow.update_states | workflow.waiting_states) \
            else 'txt-color-text'
        state_format = state_format.replace('{principal}',
                                            '<span class="{0}">{{principal}}</span>'.format(principal_class))
        # get state label
        state_label = request.registry.queryAdapter(workflow, IWorkflowStateLabel, name=state.state)
        if state_label is None:
            state_label = request.registry.queryAdapter(workflow, IWorkflowStateLabel)
        if state_label is None:
            self.state = state_format.format(state=translate(workflow.get_state_label(state.state)),
                                             principal=security.get_principal(state.state_principal).title)
        else:
            self.state = state_format.format(state=state_label.get_label(context, request, format=False),
                                             principal=security.get_principal(state.state_principal).title)
        self.state_date = translate(_("since {date}")).format(date=format_datetime(state.state_date, request=request))
        # check for links to other versions
        links = []
        visible_versions = sorted(versions.get_versions(workflow.visible_states),
                                  key=lambda x: IWorkflowState(x).version_id)
        draft_versions = sorted(versions.get_versions(workflow.update_states),
                                key=lambda x: IWorkflowState(x).version_id)
        if (state.version_id > 1) or (state.state != workflow.initial_state):
            targets = set()
            if visible_versions:
                target = visible_versions[0]
                if (target is not context) and (target not in targets):
                    links.append({'title': translate(_("access published version")),
                                  'href': absolute_url(target, request, 'admin'),
                                  'css_class': 'text-danger'})
                    targets.add(target)
            if draft_versions:
                target = draft_versions[0]
                if (target is not context) and (target not in targets):
                    links.append({'title': translate(_("access new version")),
                                  'href': absolute_url(target, request, 'admin'),
                                  'css_class': 'text-danger'})
                    targets.add(target)
            else:
                waiting_versions = sorted(versions.get_versions(workflow.waiting_states),
                                          key=lambda x: IWorkflowState(x).version_id)
                if waiting_versions:
                    target = waiting_versions[-1]
                    if (target is not context) and (target not in targets):
                        links.append({'title': translate(_("access waiting version")),
                                      'href': absolute_url(target, request, 'admin'),
                                      'css_class': 'text-danger'})
                        targets.add(target)
            if (state.version_id > 1) and (state.state not in (workflow.retired_states | workflow.archived_states)):
                retired_versions = sorted(filter(lambda x: IWorkflowState(x).version_id < state.version_id,
                                                 versions.get_versions(workflow.retired_states)),
                                          key=lambda x: IWorkflowState(x).version_id)
                if retired_versions:
                    target = retired_versions[-1]
                    if (target is not context) and (target not in targets):
                        links.append({'title': translate(_("access retired version")),
                                      'href': absolute_url(target, request, 'admin'),
                                      'css_class': 'text-danger'})
                else:
                    archived_versions = sorted(filter(lambda x: IWorkflowState(x).version_id < state.version_id,
                                                      versions.get_versions(workflow.archived_states)),
                                               key=lambda x: IWorkflowState(x).version_id)
                    if archived_versions:
                        target = archived_versions[-1]
                        if (target is not context) and (target not in targets):
                            links.append({'title': translate(_("access archived version")),
                                          'href': absolute_url(target, request, 'admin'),
                                          'css_class': 'text-danger'})
        self.versions_links = links

    @property
    def title(self):
        tool = get_parent(self.context, IBaseSharedTool)
        return II18n(tool).query_attribute('title', request=self.request)

    @reify
    def hex_oid(self):
        return ISequentialIdInfo(self.context).hex_oid

    @property
    def favorite_content(self):
        profile = IAdminProfile(self.request.principal)
        return self.hex_oid in (profile.favorites or '')


@contentprovider_config(name='content_header', context=IWfSharedContent, view=IInnerTabForm, layer=IPyAMSLayer)
class SharedContentInnerPageHeaderContentProvider(object):
    """Inner page header content provider"""

    def update(self):
        pass

    @staticmethod
    def render():
        return ''
