#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import And, Any, Eq, Or
from pyramid.decorator import reify
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import Interface, implementer
from zope.intid.interfaces import IIntIds

from pyams_catalog.query import CatalogResultSet
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION, PUBLISH_CONTENT_PERMISSION
from pyams_content.profile.interfaces import IAdminProfile
from pyams_content.shared.common import CONTENT_TYPES
from pyams_content.shared.common.interfaces import IBaseSharedTool, IManagerRestrictions, \
    ISharedTool, IWfSharedContent
from pyams_content.shared.common.interfaces.zmi import IDashboardSearchHeader, IDashboardTable, \
    IDashboardTypeColumnValue, ISharedToolDashboardTable
from pyams_content.zmi import pyams_content
from pyams_content.zmi.interfaces import IAllContentsMenu, IDashboardMenu, IMyDashboardMenu
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_security.interfaces import ISecurityManager
from pyams_sequence.interfaces import ISequentialIdInfo, ISequentialIdTarget, ISequentialIntIds
from pyams_sequence.reference import get_last_version
from pyams_skin.container import ContainerView
from pyams_skin.interfaces import IPageHeader
from pyams_skin.interfaces.container import ITableElementName
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.page import DefaultPageHeaderAdapter
from pyams_skin.table import BaseTable, I18nColumn, JsActionColumn, NameColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_template.template import template_config
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.date import SH_DATETIME_FORMAT, format_datetime
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.list import unique_iter
from pyams_utils.property import cached_property
from pyams_utils.registry import get_utility
from pyams_utils.timezone import tztime
from pyams_utils.traversing import get_parent
from pyams_viewlet.manager import viewletmanager_config
from pyams_viewlet.viewlet import viewlet_config
from pyams_workflow.interfaces import IWorkflow, IWorkflowPublicationInfo, IWorkflowState, \
    IWorkflowVersions
from pyams_zmi.interfaces.menu import IContentManagementMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import InnerAdminView


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Shared tools common adapters
#

@implementer(ISharedToolDashboardTable)
class BaseDashboardTable(BaseTable):
    """Base dashboard table"""

    _single_title = '{0} content'
    _plural_title = '{0} contents'

    sortOn = None
    dt_sort_order = 'desc'

    @property
    def title(self):
        translate = self.request.localizer.translate
        if len(self.values) < 2:
            return translate(self._single_title).format(len(self.values))
        return translate(self._plural_title).format(len(self.values))

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes['table'] = {
            'data-ams-datatable-sorting': "{0},{1}".format(len(self.columns) - 1, self.dt_sort_order),
            'data-ams-datatable-display-length': IAdminProfile(self.request.principal).table_page_length
        }
        return attributes

    @cached_property
    def values(self):
        return tuple(super().values)


@adapter_config(name='type',
                context=(Interface, IPyAMSLayer, IDashboardTable),
                provides=IColumn)
class SharedToolDashboardTypeColumn(I18nColumn, GetAttrColumn):
    """Shared tool dashboard type column"""

    _header = _("Content type")
    weight = 9

    def getValue(self, obj):
        value = self.request.registry.queryMultiAdapter((obj, self.request, self.table),
                                                        IDashboardTypeColumnValue)
        return value or '--'


@adapter_config(name='type',
                context=(ISharedTool, IPyAMSLayer, ISharedToolDashboardTable),
                provides=IColumn)
class SharedContentDashboardTypeColumn(SharedToolDashboardTypeColumn):
    """Shared tool dashboard type column"""

    _header = _("Type")


@adapter_config(name='name',
                context=(Interface, IPyAMSLayer, IDashboardTable),
                provides=IColumn)
class SharedToolDashboardNameColumn(NameColumn):
    """Shared tool dashboard name column"""

    _header = _("Title")


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, IDashboardTable),
                provides=ITableElementName)
@adapter_config(context=(IWfSharedContent, IPyAMSLayer, IDashboardTable),
                provides=ITableElementName)
class SharedToolDashboardNameAdapter(ContextRequestViewAdapter):
    """Shared tool dashboard name adapter"""

    @property
    def name(self):
        return II18n(self.context).query_attribute('title', request=self.request)


@adapter_config(name='sequence',
                context=(Interface, IPyAMSLayer, IDashboardTable),
                provides=IColumn)
class SharedToolDashboardSequenceColumn(I18nColumn, GetAttrColumn):
    """Shared tool dashboard sequence ID column"""

    _header = _("Unique ID")
    weight = 15

    def getValue(self, obj):
        target = get_parent(obj, ISequentialIdTarget)
        if target is not None:
            sequence_info = ISequentialIdInfo(obj, None)
            if sequence_info is not None:
                sequence = get_utility(ISequentialIntIds, name=target.sequence_name)
                return sequence.get_base_oid(sequence_info.oid, target.sequence_prefix)
        return '--'


@adapter_config(name='status',
                context=(Interface, IPyAMSLayer, IDashboardTable),
                provides=IColumn)
class SharedToolDashboardStatusColumn(I18nColumn, GetAttrColumn):
    """Shared tool dashboard status column"""

    _header = _("Status")
    cssClasses = {'td': 'nowrap'}
    weight = 20

    def getValue(self, obj):
        state = IWorkflowState(obj, None)
        if state is not None:
            workflow = IWorkflow(obj)
            result = self.request.localizer.translate(workflow.get_state_label(state.state))
            if state.state_urgency:
                result += ' <i class="fa fa-fw fa-exclamation-triangle txt-color-red"></i>'
            elif state.state in workflow.published_states:
                pub_info = IWorkflowPublicationInfo(obj, None)
                if (pub_info is not None) and not pub_info.is_published():
                    translate = self.request.localizer.translate
                    result += ' <i class="fa fa-fw fa-hourglass-half font-xs opacity-75 hint align-base" ' \
                              'data-ams-hint-offset="5" title="{0}"></i>'.format(
                        translate(_("Content publication start date is not passed yet")))
            return result
        return '--'


@adapter_config(name='status_date',
                context=(Interface, IPyAMSLayer, IDashboardTable),
                provides=IColumn)
class SharedToolDashboardStatusDateColumn(I18nColumn, GetAttrColumn):
    """Shared tool dashboard status date column"""

    _header = _("Status date")
    cssClasses = {'th': 'col-xs-hide col-sm-hide',
                  'td': 'col-xs-hide col-sm-hide nowrap'}
    weight = 21

    def getValue(self, obj):
        state = IWorkflowState(obj, None)
        if state is not None:
            return format_datetime(state.state_date, SH_DATETIME_FORMAT,
                                   request=self.request)
        return '--'


@adapter_config(name='version',
                context=(Interface, IPyAMSLayer, IDashboardTable),
                provides=IColumn)
class SharedToolDashboardVersionColumn(I18nColumn, GetAttrColumn):
    """Shared tool dashboard version column"""

    _header = _("Version")
    weight = 25

    def getValue(self, obj):
        state = IWorkflowState(obj, None)
        if state is not None:
            return str(state.version_id)
        return '--'


@adapter_config(name='status_principal',
                context=(Interface, IPyAMSLayer, IDashboardTable),
                provides=IColumn)
class SharedToolDashboardStatusPrincipalColumn(I18nColumn, GetAttrColumn):
    """Shared tool dashboard status principal column"""

    _header = _("Status principal")
    cssClasses = {'th': 'col-xs-hide',
                  'td': 'col-xs-hide nowrap'}
    weight = 30

    def getValue(self, obj):
        state = IWorkflowState(obj, None)
        if state is not None:
            manager = get_utility(ISecurityManager)
            return manager.get_principal(state.state_principal).title
        return '--'


@adapter_config(name='owner',
                context=(Interface, IPyAMSLayer, IDashboardTable),
                provides=IColumn)
class SharedToolDashboardOwnerColumn(I18nColumn, GetAttrColumn):
    """Shared tool dashboard owner column"""

    _header = _("Owner")
    cssClasses = {'th': 'col-xs-hide',
                  'td': 'col-xs-hide nowrap'}
    weight = 35

    def getValue(self, obj):
        try:
            owner = obj.owner
        except AttributeError:
            return '--'
        else:
            if owner:
                manager = get_utility(ISecurityManager)
                return manager.get_principal(next(iter(owner))).title
            return '--'


@adapter_config(name='modified',
                context=(Interface, IPyAMSLayer, IDashboardTable),
                provides=IColumn)
class SharedToolDashboardModifiedColumn(I18nColumn, GetAttrColumn):
    """Shared tool dashboard modified column"""

    _header = _("Last modification")
    cssClasses = {'td': 'nowrap'}
    weight = 40

    def getValue(self, obj):
        dc = IZopeDublinCore(obj, None)
        if dc is not None:
            return format_datetime(tztime(dc.modified), SH_DATETIME_FORMAT,
                                   request=self.request)
        return '--'


#
# Shared tool control panel
#

@viewlet_config(name='dashboard.menu',
                context=IBaseSharedTool, layer=IAdminLayer,
                manager=IContentManagementMenu, weight=1,
                permission=VIEW_SYSTEM_PERMISSION)
@viewletmanager_config(name='dashboard.menu',
                       layer=IAdminLayer, provides=IDashboardMenu)
@implementer(IDashboardMenu)
class SharedToolDashboardMenu(MenuItem):
    """Shared tool dashboard menu"""

    label = _("Dashboard")
    icon_class = 'fa-line-chart'
    url = '#dashboard.html'


@pagelet_config(name='dashboard.html',
                context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@template_config(template='templates/dashboard.pt', layer=IAdminLayer)
class SharedToolDashboardView(InnerAdminView):
    """Shared tool dashboard view"""

    title = _("Contents dashboard")

    def __init__(self, context, request):
        super(SharedToolDashboardView, self).__init__(context, request)
        self.tables = []
        self.tables.append(SharedToolDashboardManagerWaitingTable(self.context, self.request))
        self.tables.append(SharedToolDashboardOwnerWaitingTable(self.context, self.request))
        self.tables.append(SharedToolDashboardOwnerModifiedTable(self.context, self.request))
        for table in self.tables:
            table.hide_toolbar = True

    def update(self):
        super(SharedToolDashboardView, self).update()
        [table.update() for table in self.tables]

    @property
    def header_title(self):
        registry = self.request.registry
        adapter = registry.queryMultiAdapter((self.context, self.request, self),
                                             IDashboardSearchHeader)
        if adapter is not None:
            return adapter.title
        else:
            translate = self.request.localizer.translate
            return translate(_("SEARCH - Between all contents of type "
                               "&laquo;&nbsp;{type}&nbsp;&raquo;")).format(
                type=II18n(self.context).query_attribute('title', request=self.request)
            )


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolDashboardView),
                provides=IPageHeader)
class SharedToolDashboardHeaderAdapter(DefaultPageHeaderAdapter):
    """Shared tool properties header adapter"""

    back_url = '/admin#dashboard.html'
    back_target = None

    icon_class = 'fa fa-fw fa-line-chart'


#
# Contents waiting for manager action
#

@implementer(ISharedToolDashboardTable)
class SharedToolDashboardManagerWaitingTable(BaseDashboardTable):
    """Shared tool dashboard waiting table"""

    _single_title = _("MANAGER - {0} content waiting for your action")
    _plural_title = _("MANAGER - {0} contents waiting for your action")


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolDashboardManagerWaitingTable),
                provides=IValues)
class SharedToolDashboardManagerWaitingValues(ContextRequestViewAdapter):
    """Shared tool dashboard waiting values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        workflow = get_utility(IWorkflow, name=self.context.shared_content_workflow)
        params = Eq(catalog['parents'], intids.register(self.context)) & \
                 Any(catalog['content_type'], CONTENT_TYPES.keys()) & \
                 Any(catalog['workflow_state'], workflow.waiting_states)
        yield from filter(self.check_access,
                          unique_iter(map(lambda x: sorted(
                              IWorkflowVersions(x).get_versions(IWorkflowState(x).state),
                              key=lambda y: IZopeDublinCore(y).modified, reverse=True)[0],
                                          CatalogResultSet(
                                              CatalogQuery(catalog).query(params,
                                                                          sort_index='modified_date')))))

    def check_access(self, content):
        if self.request.has_permission(MANAGE_SITE_ROOT_PERMISSION, context=content):
            return True
        if self.request.principal.id in content.managers:
            return True
        restrictions = IManagerRestrictions(content).get_restrictions(self.request.principal)
        if restrictions is not None:
            return restrictions.check_access(content, PUBLISH_CONTENT_PERMISSION, self.request)
        return False


#
# Last owned contents waiting for action
#

@implementer(ISharedToolDashboardTable)
class SharedToolDashboardOwnerWaitingTable(BaseDashboardTable):
    """Shared tool dashboard waiting owned contents table"""

    _single_title = _("CONTRIBUTOR - {0} content waiting for action")
    _plural_title = _("CONTRIBUTOR - {0} contents waiting for action")

    dt_sort_order = 'asc'


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolDashboardOwnerWaitingTable),
                provides=IValues)
class SharedToolDashboardOwnerWaitingValues(ContextRequestViewAdapter):
    """Shared tool dashboard waiting owned contents values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        workflow = get_utility(IWorkflow, name=self.context.shared_content_workflow)
        params = Eq(catalog['parents'], intids.register(self.context)) & \
                 Any(catalog['content_type'], CONTENT_TYPES.keys()) & \
                 Any(catalog['workflow_state'], workflow.waiting_states) & \
                 Eq(catalog['workflow_principal'], self.request.principal.id)
        yield from unique_iter(
            map(lambda x: sorted(IWorkflowVersions(x).get_versions(IWorkflowState(x).state),
                                 key=lambda y: IZopeDublinCore(y).modified, reverse=True)[0],
                CatalogResultSet(CatalogQuery(catalog).query(params,
                                                             sort_index='modified_date'))))


#
# Last owned modified contents
#

@implementer(ISharedToolDashboardTable)
class SharedToolDashboardOwnerModifiedTable(BaseDashboardTable):
    """Shared tool dashboard owned modified contents table"""

    _single_title = _("CONTRIBUTOR - {0} modified content")

    @property
    def _plural_title(self):
        if len(self.values) <= 50:
            return _("CONTRIBUTOR - {0} modified contents")
        return _("CONTRIBUTOR - Last {0} modified contents")


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolDashboardOwnerModifiedTable),
                provides=IValues)
class SharedToolDashboardOwnerModifiedValues(ContextRequestViewAdapter):
    """Shared tool dashboard waiting owned contents values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        params = And(Eq(catalog['parents'], intids.register(self.context)),
                     Any(catalog['content_type'], CONTENT_TYPES.keys()),
                     Or(Eq(catalog['role:owner'], self.request.principal.id),
                        Eq(catalog['role:contributor'], self.request.principal.id)))
        yield from unique_iter(
            map(lambda x: sorted(IWorkflowVersions(x).get_versions(IWorkflowState(x).state),
                                 key=lambda y: IZopeDublinCore(y).modified, reverse=True)[0],
                CatalogResultSet(CatalogQuery(catalog).query(params,
                                                             limit=50,
                                                             sort_index='modified_date',
                                                             reverse=True))))


#
# All my contents menu
#

@viewlet_config(name='my-contents.menu',
                context=IBaseSharedTool, layer=IAdminLayer,
                manager=IContentManagementMenu, weight=5,
                permission=VIEW_SYSTEM_PERMISSION)
@viewletmanager_config(name='my-contents.menu',
                       layer=IAdminLayer, provides=IMyDashboardMenu)
@implementer(IMyDashboardMenu)
class SharedToolMyDashboardMenu(MenuItem):
    """Shared tool 'my contents' dashboard menu"""

    label = _("My contents")
    icon_class = 'fa-user'
    url = '#'


#
# My favorites
# Dashboard of favorites contents
#

@viewlet_config(name='my-favorites.menu',
                context=IBaseSharedTool, layer=IAdminLayer,
                manager=IMyDashboardMenu, weight=3,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolFavoritesMenu(MenuItem):
    """Shared tool favorites dashboard menu"""

    label = _("My favorites")
    icon_class = None
    url = '#my-favorites.html'


@implementer(ISharedToolDashboardTable)
class SharedToolFavoritesTable(BaseDashboardTable):
    """Shared tool favorites table"""

    _single_title = _("CONTRIBUTOR - {0} favorite")
    _plural_title = _("CONTRIBUTOR - {0} favorites")


@adapter_config(name='show-hide',
                context=(Interface, IPyAMSLayer, SharedToolFavoritesTable),
                provides=IColumn)
class FavoritesColumnAddRemoveColumn(JsActionColumn):
    """Favorites column add/remove column"""

    cssClasses = {'th': 'action',
                  'td': 'action switcher'}

    icon_class = 'fa fa-fw fa-star'
    icon_hint = _("Add/remove from favorites")

    url = 'PyAMS_content.profile.switchFavorite'

    weight = 1

    def __init__(self, context, request, table):
        super(FavoritesColumnAddRemoveColumn, self).__init__(context, request, table)
        self.profile = IAdminProfile(request)

    def get_icon(self, item):
        oid = ISequentialIdInfo(item).hex_oid
        icon_class = 'fa fa-fw fa-star{0}'.format(
            '' if oid in (self.profile.favorites or ()) else '-o')
        return '<i data-ams-plugins="pyams_content" ' \
               'data-ams-plugin-pyams_content-src="{path}" ' \
               'data-ams-click-handler="PyAMS_content.profile.switchFavorite" ' \
               'data-ams-stop-propagation="true" ' \
               'data-sequence-oid="{oid}" ' \
               'class="{icon_class}"></i>'.format(
            path=get_resource_path(pyams_content),
            oid=oid,
            title=self.request.localizer.translate(self.icon_hint),
            icon_class=icon_class)

    def get_url(self, item):
        return ''


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolFavoritesTable),
                provides=IValues)
class SharedToolFavoritesValues(ContextRequestViewAdapter):
    """Shared tool favorites values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        profile = IAdminProfile(self.request.principal)
        params = And(Eq(catalog['parents'], intids.register(self.context)),
                     Any(catalog['oid'], profile.favorites or ()))
        yield from unique_iter(map(get_last_version,
                                   CatalogResultSet(
                                       CatalogQuery(catalog).query(params,
                                                                   sort_index='modified_date',
                                                                   reverse=True))))


@pagelet_config(name='my-favorites.html',
                context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolFavoritesView(InnerAdminView, ContainerView):
    """Shared tool favorites view"""

    table_class = SharedToolFavoritesTable


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolFavoritesView),
                provides=IPageHeader)
class SharedToolFavoritesHeaderAdapter(DefaultPageHeaderAdapter):
    """Shared tool favorites header adapter"""

    back_url = '#dashboard.html'
    icon_class = 'fa fa-fw fa-user'

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    subtitle = _("Your favorite contents")


#
# My preparations
# Dashboard of owned and modified contents which can be updated
#

@viewlet_config(name='my-preparations.menu',
                context=IBaseSharedTool, layer=IAdminLayer,
                manager=IMyDashboardMenu, weight=5,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolPreparationsMenu(MenuItem):
    """Shared tool preparations dashboard menu"""

    label = _("My drafts")
    icon_class = None
    url = '#my-preparations.html'


@implementer(ISharedToolDashboardTable)
class SharedToolPreparationsTable(BaseDashboardTable):
    """Shared tool preparations table"""

    _single_title = _("CONTRIBUTOR - {0} content in preparation")
    _plural_title = _("CONTRIBUTOR - {0} contents in preparation")


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolPreparationsTable),
                provides=IValues)
class SharedToolPreparationsValues(ContextRequestViewAdapter):
    """Shared tool preparations values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        workflow = get_utility(IWorkflow, name=self.context.shared_content_workflow)
        params = And(Eq(catalog['parents'], intids.register(self.context)),
                     Any(catalog['content_type'], CONTENT_TYPES.keys()) &
                     Or(Eq(catalog['role:owner'], self.request.principal.id),
                        Eq(catalog['role:contributor'], self.request.principal.id)),
                     Eq(catalog['workflow_state'], workflow.initial_state))
        yield from unique_iter(CatalogResultSet(
            CatalogQuery(catalog).query(params,
                                        sort_index='modified_date',
                                        reverse=True)))


@pagelet_config(name='my-preparations.html',
                context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolPreparationsView(InnerAdminView, ContainerView):
    """Shared tool preparations view"""

    table_class = SharedToolPreparationsTable


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolPreparationsView),
                provides=IPageHeader)
class SharedToolPreparationsHeaderAdapter(DefaultPageHeaderAdapter):
    """Shared tool preparations header adapter"""

    back_url = '#dashboard.html'
    icon_class = 'fa fa-fw fa-user'

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    subtitle = _("Your prepared contents")


#
# My submissions
# Dashboard of owned and modified contents which are waiting for manager action
#

@viewlet_config(name='my-submissions.menu',
                context=IBaseSharedTool, layer=IAdminLayer,
                manager=IMyDashboardMenu, weight=7,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolSubmissionsMenu(MenuItem):
    """Shared tool submissions dashboard menu"""

    label = _("My submissions")
    icon_class = None
    url = '#my-submissions.html'


@implementer(ISharedToolDashboardTable)
class SharedToolSubmissionsTable(BaseDashboardTable):
    """Shared tool submissions table"""

    _single_title = _("CONTRIBUTOR - {0} submitted content")
    _plural_title = _("CONTRIBUTOR - {0} submitted contents")


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolSubmissionsTable),
                provides=IValues)
class SharedToolSubmissionsValues(ContextRequestViewAdapter):
    """Shared tool submissions values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        workflow = get_utility(IWorkflow, name=self.context.shared_content_workflow)
        params = And(Eq(catalog['parents'], intids.register(self.context)),
                     Any(catalog['content_type'], CONTENT_TYPES.keys()) &
                     Or(Eq(catalog['role:owner'], self.request.principal.id),
                        Eq(catalog['role:contributor'], self.request.principal.id)),
                     Any(catalog['workflow_state'], workflow.waiting_states))
        yield from unique_iter(CatalogResultSet(
            CatalogQuery(catalog).query(params,
                                        sort_index='modified_date',
                                        reverse=True)))


@pagelet_config(name='my-submissions.html',
                context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolSubmissionsView(InnerAdminView, ContainerView):
    """Shared tool submissions view"""

    table_class = SharedToolSubmissionsTable


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolSubmissionsView),
                provides=IPageHeader)
class SharedToolSubmissionsHeaderAdapter(DefaultPageHeaderAdapter):
    """Shared tool submissions header adapter"""

    back_url = '#dashboard.html'
    icon_class = 'fa fa-fw fa-user'

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    subtitle = _("Your submitted contents")


#
# My publications
# Dashboard of owned and modified contents which are published
#

@viewlet_config(name='my-publications.menu',
                context=IBaseSharedTool, layer=IAdminLayer,
                manager=IMyDashboardMenu, weight=10,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolPublicationsMenu(MenuItem):
    """Shared tool publications dashboard menu"""

    label = _("My publications")
    icon_class = None
    url = '#my-publications.html'


@implementer(ISharedToolDashboardTable)
class SharedToolPublicationsTable(BaseDashboardTable):
    """Shared tool publications table"""

    _single_title = _("CONTRIBUTOR - {0} published content")
    _plural_title = _("CONTRIBUTOR - {0} published contents")


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolPublicationsTable),
                provides=IValues)
class SharedToolPublicationsValues(ContextRequestViewAdapter):
    """Shared tool publications values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        workflow = get_utility(IWorkflow, name=self.context.shared_content_workflow)
        params = And(Eq(catalog['parents'], intids.register(self.context)),
                     Any(catalog['content_type'], CONTENT_TYPES.keys()) &
                     Or(Eq(catalog['role:owner'], self.request.principal.id),
                        Eq(catalog['role:contributor'], self.request.principal.id)),
                     Any(catalog['workflow_state'], workflow.published_states))
        yield from unique_iter(CatalogResultSet(
            CatalogQuery(catalog).query(params,
                                        sort_index='modified_date',
                                        reverse=True)))


@pagelet_config(name='my-publications.html',
                context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolPublicationsView(InnerAdminView, ContainerView):
    """Shared tool publications view"""

    table_class = SharedToolPublicationsTable


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolPublicationsView),
                provides=IPageHeader)
class SharedToolPublicationsHeaderAdapter(DefaultPageHeaderAdapter):
    """Shared tool publications header adapter"""

    back_url = '#dashboard.html'
    icon_class = 'fa fa-fw fa-user'

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    subtitle = _("Your published contents")


#
# My retired contents
# Dashboard of owned and modified contents which are retired
#

@viewlet_config(name='my-retired-contents.menu',
                context=IBaseSharedTool, layer=IAdminLayer,
                manager=IMyDashboardMenu, weight=15,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolRetiredMenu(MenuItem):
    """Shared tool retired contents dashboard menu"""

    label = _("My retired contents")
    icon_class = None
    url = '#my-retired-contents.html'


@implementer(ISharedToolDashboardTable)
class SharedToolRetiredContentsTable(BaseDashboardTable):
    """Shared tool retired contents table"""

    _single_title = _("CONTRIBUTOR - {0} retired content")
    _plural_title = _("CONTRIBUTOR - {0} retired contents")


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolRetiredContentsTable),
                provides=IValues)
class SharedToolRetiredContentsValues(ContextRequestViewAdapter):
    """Shared tool retired contents values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        workflow = get_utility(IWorkflow, name=self.context.shared_content_workflow)
        params = And(Eq(catalog['parents'], intids.register(self.context)),
                     Any(catalog['content_type'], CONTENT_TYPES.keys()) &
                     Or(Eq(catalog['role:owner'], self.request.principal.id),
                        Eq(catalog['role:contributor'], self.request.principal.id)),
                     Any(catalog['workflow_state'], workflow.retired_states))
        yield from unique_iter(CatalogResultSet(
            CatalogQuery(catalog).query(params,
                                        sort_index='modified_date',
                                        reverse=True)))


@pagelet_config(name='my-retired-contents.html',
                context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolRetiredContentsView(InnerAdminView, ContainerView):
    """Shared tool retired contents view"""

    table_class = SharedToolRetiredContentsTable


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolRetiredContentsView),
                provides=IPageHeader)
class SharedToolRetiredContentsHeaderAdapter(DefaultPageHeaderAdapter):
    """Shared tool retired contents header adapter"""

    back_url = '#dashboard.html'
    icon_class = 'fa fa-fw fa-user'

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    subtitle = _("Your retired contents")


#
# My archived contents
# Dashboard of owned and modified contents which are archived
#

@viewlet_config(name='my-archived-contents.menu',
                context=IBaseSharedTool, layer=IAdminLayer,
                manager=IMyDashboardMenu, weight=20,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolArchivedMenu(MenuItem):
    """Shared tool archived contents dashboard menu"""

    label = _("My archived contents")
    icon_class = None
    url = '#my-archived-contents.html'


@implementer(ISharedToolDashboardTable)
class SharedToolArchivedContentsTable(BaseDashboardTable):
    """Shared tool archived contents table"""

    _single_title = _("CONTRIBUTOR - {0} archived content")
    _plural_title = _("CONTRIBUTOR - {0} archived contents")


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolArchivedContentsTable),
                provides=IValues)
class SharedToolArchivedContentsValues(ContextRequestViewAdapter):
    """Shared tool archived contents values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        principal_id = self.request.principal.id
        workflow = get_utility(IWorkflow, name=self.context.shared_content_workflow)
        params = And(Eq(catalog['parents'], intids.register(self.context)),
                     Any(catalog['content_type'], CONTENT_TYPES.keys()) &
                     Or(Eq(catalog['role:owner'], principal_id),
                        Eq(catalog['role:contributor'], principal_id)),
                     Any(catalog['workflow_state'], workflow.archived_states))
        yield from unique_iter(map(lambda x: sorted((
            version for version in IWorkflowVersions(x).get_versions(workflow.archived_states)
            if principal_id in (version.owner | version.contributors)
        ), key=lambda x: IWorkflowState(x).version_id,
           reverse=True)[0],
           CatalogResultSet(CatalogQuery(catalog).query(params,
                                                        sort_index='modified_date',
                                                        reverse=True))))


@pagelet_config(name='my-archived-contents.html',
                context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolArchivedContentsView(InnerAdminView, ContainerView):
    """Shared tool archived contents view"""

    table_class = SharedToolArchivedContentsTable


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolArchivedContentsView),
                provides=IPageHeader)
class SharedToolArchivedContentsHeaderAdapter(DefaultPageHeaderAdapter):
    """Shared tool archived contents header adapter"""

    back_url = '#dashboard.html'
    icon_class = 'fa fa-fw fa-user'

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    subtitle = _("Your archived contents")


#
# All interventions
#

@viewlet_config(name='all-interventions.menu',
                context=IBaseSharedTool, layer=IAdminLayer,
                manager=IContentManagementMenu, weight=10,
                permission=VIEW_SYSTEM_PERMISSION)
@viewletmanager_config(name='all-interventions.menu',
                       layer=IAdminLayer, provides=IAllContentsMenu)
@implementer(IAllContentsMenu)
class SharedToolAllContentsMenu(MenuItem):
    """Shared tool 'all contents' dashboard menu"""

    label = _("Other interventions")
    css_class = 'open'
    icon_class = 'fa-pencil-square'
    url = '#'


#
# Last publications
# Dashboard of all published contents
#

@viewlet_config(name='all-publications.menu',
                context=IBaseSharedTool, layer=IAdminLayer,
                manager=IAllContentsMenu, weight=10,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolAllPublicationsMenu(MenuItem):
    """Shared tool published contents dashboard menu"""

    label = _("Last publications")
    icon_class = None
    url = '#all-publications.html'


@implementer(ISharedToolDashboardTable)
class SharedToolAllPublicationsTable(BaseDashboardTable):
    """Shared tool published contents table"""

    _single_title = _("CONTRIBUTORS - {0} published content")

    @property
    def _plural_title(self):
        if len(self.values) <= 50:
            return _("CONTRIBUTORS - Last {0} published contents")
        return _("CONTRIBUTORS - Last published contents (in the limit of 50)")


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolAllPublicationsTable),
                provides=IValues)
class SharedToolAllPublicationsValues(ContextRequestViewAdapter):
    """Shared tool published contents values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        workflow = get_utility(IWorkflow, name=self.context.shared_content_workflow)
        params = And(Eq(catalog['parents'], intids.register(self.context)),
                     Any(catalog['content_type'], CONTENT_TYPES.keys()) &
                     Any(catalog['workflow_state'], workflow.published_states))
        yield from unique_iter(CatalogResultSet(
            CatalogQuery(catalog).query(params,
                                        limit=50,
                                        sort_index='modified_date',
                                        reverse=True)))


@pagelet_config(name='all-publications.html',
                context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolAllPublicationsView(InnerAdminView, ContainerView):
    """Shared tool published contents view"""

    table_class = SharedToolAllPublicationsTable


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolAllPublicationsView),
                provides=IPageHeader)
class SharedToolAllPublicationsHeaderAdapter(DefaultPageHeaderAdapter):
    """Shared tool published contents header adapter"""

    back_url = '#dashboard.html'
    icon_class = 'fa fa-fw fa-pencil-square'

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    subtitle = _("Last published contents")


#
# Last updates
# Dashboard of all updated contents
#

@viewlet_config(name='all-updates.menu',
                context=IBaseSharedTool, layer=IAdminLayer,
                manager=IAllContentsMenu, weight=20,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolAllUpdatesMenu(MenuItem):
    """Shared tool updated contents dashboard menu"""

    label = _("Last updates")
    icon_class = None
    url = '#all-updates.html'


@implementer(ISharedToolDashboardTable)
class SharedToolAllUpdatesTable(BaseDashboardTable):
    """Shared tool updated contents table"""

    _single_title = _("CONTRIBUTORS - {0} updated content")

    @property
    def _plural_title(self):
        if len(self.values) <= 50:
            return _("CONTRIBUTORS - Last {0} updated contents")
        return _("CONTRIBUTORS - Last updated contents (in the limit of 50)")


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolAllUpdatesTable),
                provides=IValues)
class SharedToolAllUpdatesValues(ContextRequestViewAdapter):
    """Shared tool updated contents values adapter"""

    @property
    def values(self):
        intids = get_utility(IIntIds)
        catalog = get_utility(ICatalog)
        params = Eq(catalog['parents'], intids.register(self.context)) & \
                 Any(catalog['content_type'], CONTENT_TYPES.keys())
        yield from unique_iter(CatalogResultSet(
            CatalogQuery(catalog).query(params,
                                        limit=50,
                                        sort_index='modified_date',
                                        reverse=True)))


@pagelet_config(name='all-updates.html',
                context=IBaseSharedTool, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class SharedToolAllUpdatesView(InnerAdminView, ContainerView):
    """Shared tool updated contents view"""

    table_class = SharedToolAllUpdatesTable


@adapter_config(context=(IBaseSharedTool, IPyAMSLayer, SharedToolAllUpdatesView),
                provides=IPageHeader)
class SharedToolAllUpdatesHeaderAdapter(DefaultPageHeaderAdapter):
    """Shared tool updated contents header adapter"""

    back_url = '#dashboard.html'
    icon_class = 'fa fa-fw fa-pencil-square'

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    subtitle = _("Last updated contents")
