#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from pyramid.events import subscriber
from zope.container.contained import Contained
from zope.container.ordered import OrderedContainer
from zope.interface import implementer
from zope.lifecycleevent.interfaces import IObjectAddedEvent
from zope.location.interfaces import ISublocations
from zope.schema import getFieldsInOrder
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary
from zope.traversing.interfaces import ITraversable

from pyams_content.component.extfile.interfaces import IExtFileContainerTarget
from pyams_content.component.links.interfaces import ILinkContainerTarget
from pyams_content.component.paragraph.interfaces import IParagraphContainerTarget
from pyams_content.component.theme.interfaces import ITagsInfo, ITagsTarget, IThemesInfo, \
    IThemesTarget
from pyams_content.features.json import IJSONConverter, JSONBaseConverter
from pyams_content.interfaces import MANAGE_TOOL_PERMISSION
from pyams_content.reference.pictograms import IPictogramTable
from pyams_content.shared.common import IWfSharedContentFactory
from pyams_content.shared.common.interfaces import ISharedContentFactory, ISharedTool
from pyams_content.shared.common.interfaces.types import ALL_DATA_TYPES_VOCABULARY, \
    DATA_MANAGER_ANNOTATION_KEY, DATA_SUBTYPES_VOCABULARY, DATA_TYPES_VOCABULARY, \
    DATA_TYPE_FIELDS_VOCABULARY, IBaseDataType, IDataType, ISubType, ITypedDataManager, \
    ITypedSharedTool, IWfTypedSharedContent
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n
from pyams_sequence.reference import get_reference_target
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utilities_for, query_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_utils.vocabulary import vocabulary_config

__docformat__ = 'restructuredtext'

from pyams_content import _


class BaseDataType(Persistent, Contained):
    """Base data type"""

    label = FieldProperty(IBaseDataType['label'])
    source_folder = FieldProperty(IBaseDataType['source_folder'])
    navigation_label = FieldProperty(IBaseDataType['navigation_label'])
    facets_label = FieldProperty(IBaseDataType['facets_label'])
    facets_type_label = FieldProperty(IBaseDataType['facets_type_label'])
    backoffice_label = FieldProperty(IBaseDataType['backoffice_label'])
    color = FieldProperty(IBaseDataType['color'])
    pictogram = FieldProperty(IBaseDataType['pictogram'])
    pictogram_on = FieldProperty(IBaseDataType['pictogram_on'])
    pictogram_off = FieldProperty(IBaseDataType['pictogram_off'])

    def get_source_folder(self):
        if self.source_folder is not None:
            return get_reference_target(self.source_folder)

    def get_pictogram(self, name=''):
        table = query_utility(IPictogramTable)
        if table is not None:
            name = 'pictogram{}{}'.format(* ('_', name) if name else ('', ''))
            pictogram_name = getattr(self, name, None)
            if pictogram_name:
                return table.get(pictogram_name)


@adapter_config(required=(IBaseDataType, IPyAMSLayer),
                provides=IJSONConverter)
class BaseDataTypeJSONConverter(JSONBaseConverter):
    """Base data type JSON converter"""

    def to_json(self, params=None):
        """JSON converter"""
        context = self.context
        request = self.request
        result = {
            'name': context.__name__,
            'label': II18n(context).query_attribute('label',
                                                    lang=params.get('lang'))
        }
        label = II18n(context).query_attribute('navigation_label',
                                               lang=params.get('lang'))
        if label:
            result['navigation_label'] = label
        if context.color:
            result['color'] = '#{}'.format(context.color)
        pictograms = {}
        for name in ('', 'on', 'off'):
            pictogram = context.get_pictogram(name)
            if pictogram is not None:
                image = II18n(pictogram).query_attribute('image', request=request)
                if image:
                    pictograms[name or 'default'] = {
                        'content_type': image.content_type,
                        'src': absolute_url(image, request)
                    }
        if pictograms:
            result['pictograms'] = pictograms
        return result


@implementer(ISubType, IParagraphContainerTarget, IExtFileContainerTarget, ILinkContainerTarget)
class SubType(BaseDataType):
    """Data sub-type persistent class"""


@implementer(IDataType, IParagraphContainerTarget, IExtFileContainerTarget,
             ILinkContainerTarget, ITagsTarget, IThemesTarget)
class DataType(BaseDataType, OrderedContainer):
    """Data type persistent class"""

    display_as_tag = FieldProperty(IDataType['display_as_tag'])
    field_names = FieldProperty(IDataType['field_names'])


@factory_config(ITypedDataManager)
class TypedDataManager(OrderedContainer):
    """Data types container persistent class"""


@adapter_config(required=IBaseDataType,
                provides=IFormContextPermissionChecker)
class BaseDatatypePermissionChecker(ContextAdapter):
    """Base data type permission checker"""

    edit_permission = MANAGE_TOOL_PERMISSION


@implementer(ITypedSharedTool)
class TypedSharedToolMixin(object):
    """Typed shared tool"""

    shared_content_types_fields = None


@adapter_config(required=ITypedSharedTool,
                provides=ITypedDataManager)
def typed_shared_tool_data_manager_factory(context):
    """Types shared tool data manager factory"""
    return get_annotation_adapter(context, DATA_MANAGER_ANNOTATION_KEY, ITypedDataManager,
                                  name='++types++')


@adapter_config(name='types',
                context=ITypedSharedTool,
                provides=ITraversable)
class TypedSharedToolTypesNamespace(ContextAdapter):
    """Typed shared tool ++types++ namespace"""

    def traverse(self, name, furtherpath=None):
        return ITypedDataManager(self.context)


@adapter_config(name='types',
                required=ITypedSharedTool,
                provides=ISublocations)
class TypedSharedToolSublocations(ContextAdapter):
    """Typed shared tool sublocations adapter"""

    def sublocations(self):
        return ITypedDataManager(self.context).values()


#
# Typed shared content
#

@implementer(IWfTypedSharedContent)
class WfTypedSharedContentMixin(object):
    """Typed shared content"""

    data_type = FieldProperty(IWfTypedSharedContent['data_type'])

    def get_data_type(self):
        if not self.data_type:
            return None
        tool = get_parent(self, ITypedSharedTool)
        if tool is not None:
            manager = ITypedDataManager(tool)
            return manager.get(self.data_type)

    @property
    def field_names(self):
        data_type = self.get_data_type()
        if data_type is not None:
            return data_type.field_names


@subscriber(IObjectAddedEvent, context_selector=IWfTypedSharedContent)
def handle_added_typed_shared_content(event):
    """Automatically assign themes for newly created contents"""
    content = event.object
    data_type = content.get_data_type()
    if data_type is not None:
        if ITagsTarget.providedBy(content):
            tags_info = ITagsInfo(content)
            if not tags_info.tags:
                tags_info.tags = ITagsInfo(data_type).tags
        if IThemesTarget.providedBy(content):
            themes_info = IThemesInfo(content)
            if not themes_info.themes:  # don't remove previous themes!
                themes_info.themes = IThemesInfo(data_type).themes


#
# Data types vocabularies
#

@vocabulary_config(name=ALL_DATA_TYPES_VOCABULARY)
class AllTypedSharedToolDataTypesVocabulary(SimpleVocabulary):
    """Vocabulary consolidating all data types"""

    def __init__(self, context):
        terms = []
        request = check_request()
        for name, tool in get_utilities_for(ISharedTool):
            if not name:
                continue
            manager = ITypedDataManager(tool, None)
            if manager is not None:
                terms.extend([
                    SimpleTerm(datatype.__name__,
                               title=II18n(datatype).query_attribute('label',
                                                                     request=request))
                    for datatype in manager.values()
                ])
        terms.sort(key=lambda x: x.title)
        super().__init__(terms)

    def getTermByToken(self, token):
        try:
            return super().getTermByToken(token)
        except LookupError:
            request = check_request()
            translate = request.localizer.translate
            return SimpleTerm(token,
                              title=translate(_("-- missing value ({}) --")).format(token))


def get_all_data_types(request):
    """Get list of all registered data types as JSON object"""
    results = []
    for name, tool in sorted(get_utilities_for(ISharedTool),
                             key=lambda x: II18n(x[1]).query_attribute('title', request=request)):
        if not name:
            continue
        manager = ITypedDataManager(tool, None)
        if manager is not None:
            terms = [
                {
                    'id': datatype.__name__,
                    'text': II18n(datatype).query_attribute('label', request=request)
                }
                for datatype in manager.values()
            ]
            content_factory = IWfSharedContentFactory(ISharedContentFactory(tool))
            results.append({
                'text': request.localizer.translate(content_factory.content_name),
                'disabled': True,
                'children': terms
            })
    return results


@vocabulary_config(name=DATA_TYPES_VOCABULARY)
class TypedSharedToolDataTypesVocabulary(SimpleVocabulary):
    """Typed shared tool data types vocabulary"""

    def __init__(self, context):
        terms = []
        parent = get_parent(context, ITypedSharedTool)
        if parent is not None:
            request = check_request()
            manager = ITypedDataManager(parent)
            terms = [
                SimpleTerm(datatype.__name__,
                           title=II18n(datatype).query_attribute('label',
                                                                 request=request))
                for datatype in manager.values()
            ]
        super().__init__(terms)


@vocabulary_config(name=DATA_SUBTYPES_VOCABULARY)
class TypedSharedToolDataSubtypesVocabulary(SimpleVocabulary):
    """Typed shared tool data subtypes vocabulary"""

    def __init__(self, context):
        terms = []
        if IWfTypedSharedContent.providedBy(context):
            parent = get_parent(context, ITypedSharedTool)
            if parent is not None:
                request = check_request()
                datatype = ITypedDataManager(parent).get(context.data_type)
                terms = [
                    SimpleTerm(subtype.__name__,
                               title=II18n(subtype).query_attribute('label',
                                                                    request=request))
                    for subtype in datatype.values()
                ]
        super().__init__(terms)


@vocabulary_config(name=DATA_TYPE_FIELDS_VOCABULARY)
class TypedSharedToolDataTypesFieldsVocabulary(SimpleVocabulary):
    """Typed shared tool data types fields vocabulary"""

    def __init__(self, context):
        terms = []
        parent = get_parent(context, ITypedSharedTool)
        if (parent is not None) and parent.shared_content_types_fields:
            request = check_request()
            translate = request.localizer.translate
            terms = [
                SimpleTerm(name, title=translate(field.title))
                for name, field in getFieldsInOrder(parent.shared_content_types_fields)
            ]
        super().__init__(terms)
