#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from zope.container.constraints import containers, contains
from zope.container.interfaces import IContainer
from zope.interface import Attribute, Interface, implementer
from zope.schema import Bool, Choice, Text, TextLine

from pyams_content.interfaces import API_ROLE, CONTENT_PUBLICATION_DATE_ORDER, CONTRIBUTOR_ROLE, GUEST_ROLE, \
    IBaseContent, MANAGER_ROLE, MANAGE_CONTENT_PERMISSION, OWNER_ROLE, PILOT_ROLE, READER_ROLE, WEBMASTER_ROLE
from pyams_i18n.schema import I18nTextField, I18nTextLineField
from pyams_portal.interfaces import DESIGNER_ROLE, IPortalContext
from pyams_security.schema import Principal, PrincipalsSet
from pyams_utils.schema import TextLineListField
from pyams_workflow.interfaces import IWorkflowManagedContent

__docformat__ = 'restructuredtext'

from pyams_content import _


class IDeletableElement(Interface):
    """Deletable element interface"""

    def is_deletable(self):
        """Check to know if a site element can be deleted"""


class IBaseContentManagerRoles(Interface):
    """Shared tool roles interface"""

    api_clients = PrincipalsSet(title=_("API clients"),
                                description=_("This principals are allowed to use public APIs"),
                                role_id=API_ROLE,
                                required=False)

    webmasters = PrincipalsSet(title=_("Webmasters"),
                               description=_("Webmasters can handle all contents, including "
                                             "published ones"),
                               role_id=WEBMASTER_ROLE,
                               required=False)

    pilots = PrincipalsSet(title=_("Pilots"),
                           description=_("Pilots can handle tool configuration, manage access "
                                         "rules, grant users roles and manage managers "
                                         "restrictions"),
                           role_id=PILOT_ROLE,
                           required=False)

    managers = PrincipalsSet(title=_("Managers"),
                             description=_("Managers can handle main operations in tool's "
                                           "workflow, like publish or retire contents"),
                             role_id=MANAGER_ROLE,
                             required=False)

    contributors = PrincipalsSet(title=_("Contributors"),
                                 description=_("Contributors are users which are allowed to "
                                               "create new contents"),
                                 role_id=CONTRIBUTOR_ROLE,
                                 required=False)

    designers = PrincipalsSet(title=_("Designers"),
                              description=_("Designers are users which are allowed to manage "
                                            "presentation templates"),
                              role_id=DESIGNER_ROLE,
                              required=False)


class ISharedSite(IBaseContent, IDeletableElement):
    """Shared site interface"""

    containers('pyams_content.root.interfaces.ISiteRoot')


class ISharedToolContainer(IBaseContent, IContainer):
    """Shared tools container"""

    containers('pyams_content.root.interfaces.ISiteRoot')
    contains('.ISharedTool')


class IBaseSharedTool(IBaseContent, IContainer):
    """Base shared tool interface"""

    containers(ISharedToolContainer)

    shared_content_menu = Attribute("Boolean flag indicating if tool is displayed into 'Shared "
                                    "contents' or Shared tools' menu")

    shared_content_workflow = Choice(title=_("Workflow name"),
                                     description=_("Name of workflow utility used to manage tool "
                                                   "contents"),
                                     vocabulary="PyAMS workflows",
                                     default="PyAMS default workflow")

    label = I18nTextLineField(title=_("Single content label"),
                              description=_("This label can be used to tag content type of a single content "
                                            "in place of shared tool title"),
                              required=False)

    navigation_label = I18nTextLineField(title=_("Navigation label"),
                                         description=_("Label used for navigation entries"),
                                         required=False)

    facets_label = I18nTextLineField(title=_("Facets label"),
                                     description=_("Label used for the facets of search engines, instead "
                                                   "of the standard label"),
                                     required=False)
    
    facets_type_label = I18nTextLineField(title=_("Facets type label"),
                                          description=_("Label used for the facets of search engine, instead "
                                                        "of the standard label, when facets are configured in "
                                                        "\"content-type\" mode"),
                                          required=False)

    backoffice_label = I18nTextLineField(title=_("Dashboard label"),
                                         description=_("Label used for the presentation of dashboards"),
                                         required=False)


SHARED_TOOL_WORKFLOW_STATES_VOCABULARY = 'PyAMS workflow states'


class ISharedTool(IBaseSharedTool):
    """Shared tool interface"""

    contains('.ISharedContent')

    shared_content_type = Attribute("Shared data content type")
    shared_content_factory = Attribute("Shared data factory")

    def create(self, params, request=None):
        """Create a new content from given params"""

    def find(self, params, request=None,
             sort_index=CONTENT_PUBLICATION_DATE_ORDER, reverse=True,
             limit=999):
        """Find contents matching given params"""


class ISharedToolPortalContext(ISharedTool, IPortalContext):
    """Shared tool with portal context"""


class ISharedToolRoles(IBaseContentManagerRoles):
    """Shared tool roles"""


class IWfSharedContentCreator(Interface):
    """Shared content creator interface"""

    def update_content(self, params, request=None):
        """Update a new content with values extracted from params"""


class IWfSharedContentUpdater(Interface):
    """Shared content updater interface"""

    def update_content(self, params, request=None):
        """Updated existing content with values extracted from params"""


class IWfSharedContentFinder(Interface):
    """Shared content finder interface"""

    def get_results(self, params, request=None,
                    sort_index=CONTENT_PUBLICATION_DATE_ORDER, reverse=True,
                    limit=999):
        """Get query results from provided params"""


class IWfSharedContentFinderParams(Interface):
    """Shared content finder params getter interface"""

    def get_params(self, query, params, request=None, **kwargs):
        """Get query filter from provided params"""


class IWfSharedContentFilter(Interface):
    """Shared content filter"""

    def get_filter(self, search, params, request=None, **kwargs):
        """Add filter to provided search from params"""


class IWfSharedContent(IBaseContent):
    """Shared content interface"""

    content_type = Attribute("Content data type")
    content_name = Attribute("Content name")

    content_url = TextLine(title=_("Content URL"),
                           description=_("URL used to access this content; this is important for "
                                         "SEO and should include most important words describing "
                                         "content; spaces and underscores will be automatically "
                                         "replaced by hyphens"),
                           required=True)

    handle_content_url = Attribute("Static boolean value to specify if content URL is supported by "
                                   "this content type")

    creator = Principal(title=_("Version creator"),
                        description=_("Name of content's version creator. "
                                      "The creator of the first version is also it's owner."),
                        required=True)

    first_owner = Principal(title=_("First owner"),
                            description=_("Name of content's first version owner"),
                            required=True,
                            readonly=True)

    creation_label = TextLine(title=_("Version creation"),
                              readonly=True)

    modifiers = PrincipalsSet(title=_("Version modifiers"),
                              description=_("List of principals who modified this content"),
                              required=False)

    last_modifier = Principal(title=_("Last modifier"),
                              description=_("Last principal who modified this content"),
                              required=False)

    last_update_label = TextLine(title=_("Last update"),
                                 readonly=True)

    header = I18nTextField(title=_("Header"),
                           description=_("Content's header is generally displayed in page header"),
                           required=False)

    handle_header = Attribute("Static boolean value to specify if header is supported by this "
                              "content type")

    description = I18nTextField(title=_("Meta-description"),
                                description=_("The content's description is 'hidden' into HTML's "
                                              "page headers; but it can be seen, for example, in "
                                              "some search engines results as content's "
                                              "description; if description is empty, content's "
                                              "header will be used."),
                                required=False)

    handle_description = Attribute("Static boolean value to specify if description is supported by "
                                   "this content type")

    keywords = TextLineListField(title=_("Keywords"),
                                 description=_("They will be included into HTML pages metadata"),
                                 required=False)

    notepad = Text(title=_("Notepad"),
                   description=_("Internal information to be known about this content"),
                   required=False)

    def update_from_json(self, params, request=None):
        """Update content from given params"""

    def to_json(self, params=None):
        """Convert content to JSON"""


class IBaseContentPortalContext(IPortalContext):
    """Content portal context interface"""


class IWfSharedContentPortalContext(IWfSharedContent, IBaseContentPortalContext):
    """Shared content with portal support"""


class IWfSharedContentFactory(Interface):
    """Shared content factory interface"""


class IWfSharedContentRoles(Interface):
    """Shared content roles"""

    owner = PrincipalsSet(title=_("Content owner"),
                          description=_("The owner is the creator of content's first version, "
                                        "except if it was transferred afterwards to another owner"),
                          role_id=OWNER_ROLE,
                          required=True,
                          max_length=1)

    managers = PrincipalsSet(title=_("Managers"),
                             description=_("Managers can handle main operations in tool's "
                                           "workflow, like publish or retire contents"),
                             role_id=MANAGER_ROLE,
                             required=False)

    contributors = PrincipalsSet(title=_("Contributors"),
                                 description=_("Contributors are users which are allowed to update "
                                               "this content in addition to it's owner"),
                                 role_id=CONTRIBUTOR_ROLE,
                                 required=False)

    designers = PrincipalsSet(title=_("Designers"),
                              description=_("Designers are users which are allowed to manage "
                                            "presentation templates"),
                              role_id=DESIGNER_ROLE,
                              required=False)

    readers = PrincipalsSet(title=_("Readers"),
                            description=_("Readers are users which are asked to verify and comment "
                                          "contents before they are published"),
                            role_id=READER_ROLE,
                            required=False)

    guests = PrincipalsSet(title=_("Guests"),
                           description=_("Guests are users which are allowed to view contents "
                                         "with restricted access"),
                           role_id=GUEST_ROLE,
                           required=False)


class ISharedContent(IWorkflowManagedContent):
    """Workflow managed shared content interface"""

    visible_version = Attribute("Link to actually visible version")


class ISharedContentFactory(Interface):
    """Workflow managed shared content factory interface"""


CONTENT_TYPES_VOCABULARY = 'PyAMS content types'

SHARED_CONTENT_TYPES_VOCABULARY = 'PyAMS shared content types'


#
# Shared content notification event
#

class IWfSharedContentNotificationEvent(Interface):
    """Shared content notification source event interface"""

    object = Attribute("Source object")
    request = Attribute("Source request")


@implementer(IWfSharedContentNotificationEvent)
class WfSharedContentNotificationEvent:
    """Shared content notification source event"""

    def __init__(self, object, request):
        self.object = object
        self.request = request


#
# Generic restrictions interfaces
#

class IRestrictionInfo(Interface):
    """User restriction base interface"""

    principal_id = Principal(title=_("Principal ID"),
                             required=True)

    restriction_interface = Attribute("Restrictions interface")


class IRestrictions(Interface):
    """Restrictions manager interface"""

    restrictions_key = Attribute("Restrictions annotations key")
    restrictions_factory_interface = Attribute("Restrictions factory")

    def get_restrictions(self, principal, create_if_none=False):
        """Get manager restrictions for given principal"""

    def new_restrictions(self, principal):
        """Create new manager restrictions"""

    def set_restrictions(self, principal, restrictions=None):
        """Set manager restrictions for given principal"""

    def drop_restrictions(self, principal):
        """Drop manager restrictions for given principal"""


class IRestrictionsFactory(Interface):
    """Restrictions factory interface"""


#
# Shared tool contributor security restrictions
#

CONTRIBUTOR_RESTRICTIONS_KEY = 'pyams_content.contributor.restrictions'


class IContributorRestrictionInfo(IRestrictionInfo):
    """Shared content contributor restrictions"""

    publication_checks = Bool(title=_("Publication checks"),
                              description=_("If 'yes', this contributor will have to confirm that "
                                            "contents have been previewed and checked before "
                                            "asking for publication"),
                              required=False,
                              default=True)

    owners = PrincipalsSet(title=_("Substitute for"),
                           description=_("Contributor will have access to contents owned by these "
                                         "principals"),
                           required=False)

    def check_access(self, context, permission=MANAGE_CONTENT_PERMISSION, request=None):
        """Check if principal is granted access to given context"""


class IContributorRestrictions(IRestrictions):
    """Contributor restrictions interface"""


class IContributorRestrictionsFactory(IRestrictionsFactory):
    """Contributor restrictions factory interface"""


SHARED_TOOL_CONTRIBUTORS_VOCABULARY = 'PyAMS shared tool contributors'


#
# Shared tool manager security restrictions
#

MANAGER_RESTRICTIONS_KEY = 'pyams_content.manager.restrictions'


class IManagerRestrictionInfo(IRestrictionInfo):
    """Shared content manager restrictions"""

    publication_checks = Bool(title=_("Publication checks"),
                              description=_("If 'yes', this manager will have to confirm that "
                                            "contents have been previewed and checked before "
                                            "publishing a content"),
                              required=False,
                              default=True)

    restricted_contents = Bool(title=_("Restricted contents"),
                               description=_("If 'yes', this manager will get restricted access "
                                             "to manage contents based on selected settings"),
                               required=False,
                               default=True)

    owners = PrincipalsSet(title=_("Selected owners"),
                           description=_("Manager will have access to contents owned by these "
                                         "principals"),
                           required=False)

    def check_access(self, context, permission=MANAGE_CONTENT_PERMISSION, request=None):
        """Check if principal is granted access to given content"""


class IManagerRestrictions(IRestrictions):
    """Manager restrictions interface"""


class IManagerRestrictionsFactory(IRestrictionsFactory):
    """Manager restrictions factory interface"""
