#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

import enum

from colander import Enum, Integer, MappingSchema, SchemaNode, SequenceSchema, String, drop

from pyams_content.component.gallery.api.schema import GalleryInfo
from pyams_content.component.illustration.api.schema import IllustrationInfo
from pyams_content.component.paragraph.api.schema import ParagraphsListInfo
from pyams_content.component.theme.api.schema import TermsListInfo
from pyams_content.reference.pictograms.api.schema import PictogramsMapping
from pyams_content.workflow.interfaces.api import WorkflowPublicationInfo
from pyams_utils.rest import BaseResponseSchema


class DisplayName(enum.Enum):
    """Image display enumeration"""
    xs = 'xs'
    sm = 'sm'
    md = 'md'
    lg = 'lg'
    portrait = 'portrait'
    square = 'square'
    pano = 'pano'
    banner = 'banner'


class DataTypeInfo(MappingSchema):
    """Shared data type information schema"""
    name = SchemaNode(String(),
                      description="Data type name")
    label = SchemaNode(String(),
                       description="Data type label")
    navigation_label = SchemaNode(String(),
                                  description="Data type navigation label",
                                  missing=drop)
    color = SchemaNode(String(),
                       description="Data type base color in #RRGGBB format",
                       missing=drop)
    pictograms = PictogramsMapping(description="Pictograms information",
                                   missing=drop)


class DataTypesList(SequenceSchema):
    """Shared data types list"""
    content = DataTypeInfo()


class DataTypesResult(BaseResponseSchema):
    """"""
    results = DataTypesList(description="Data types list")


class DataTypesResponse(MappingSchema):
    """Shared data types response"""
    body = DataTypesResult(description="Data types response")


class SharedContentSpecificitiesInfo(MappingSchema):
    """Shared content specificities schema"""


class SharedContentInfo(MappingSchema):
    """Shared content mapping"""
    oid = SchemaNode(String(),
                     description="Content unique ID")
    title = SchemaNode(String(),
                       description="Content title")
    data_type = SchemaNode(String(),
                           description="Content data type name",
                           missing=drop)
    header = SchemaNode(String(),
                        description="Content header",
                        missing=drop)
    description = SchemaNode(String(),
                             description="Content description",
                             missing=drop)
    content_url = SchemaNode(String(),
                             description="Text used in content canonical URL",
                             missing=drop)
    illustration = IllustrationInfo(description="Illustration information",
                                    missing=drop)
    gallery = GalleryInfo(description="Medias gallery info",
                          missing=drop)
    workflow = WorkflowPublicationInfo(description="Workflow information",
                                       missing=drop)
    specificities = SharedContentSpecificitiesInfo(description="Content specificities",
                                                   missing=drop)
    paragraphs = ParagraphsListInfo(description="Paragraphs list",
                                    missing=drop)
    tags = TermsListInfo(description="Content tags",
                         missing=drop)
    themes = TermsListInfo(description="Content themes",
                           missing=drop)
    collections = TermsListInfo(description="Content collections",
                                missing=drop)


class BaseContentInfoSearchQuery(MappingSchema):
    """Base content info search query"""
    display_name = SchemaNode(Enum(DisplayName),
                              description="Images display name",
                              missing=drop)
    display_size = SchemaNode(String(),
                              description="Images display size, in pixels (syntax: 'w640' for width, 'h480' for "
                                          "height, or '640x480' for both",
                              missing=drop)


class BaseContentSearchQuery(BaseContentInfoSearchQuery):
    """Base content search query"""
    included = SchemaNode(String(),
                          description="Comma separated list of extensions to include into JSON results output",
                          missing=drop)
    excluded = SchemaNode(String(),
                          description="Comma separated list of extensions excluded from JSON results output",
                          missing=drop)


class ContentSearchQuery(BaseContentSearchQuery):
    """Content search schema"""
    data_type = SchemaNode(String(),
                           description="Select only provided data type",
                           missing=drop)
    age_limit = SchemaNode(Integer(),
                           description="Maximum days from content publication date, in days",
                           missing=drop)


class ContentSearchRequest(MappingSchema):
    """Content search request"""
    querystring = ContentSearchQuery()


class ContentResultsList(SequenceSchema):
    """Content results list"""
    content = SharedContentInfo()


class ContentSearchResult(BaseResponseSchema):
    """Content search result"""
    results = ContentResultsList(description="Content search results",
                                 missing=drop)


class ContentSearchResponse(MappingSchema):
    """Content search response"""
    body = ContentSearchResult()


class ContentGetterRequest(MappingSchema):
    """Content getter request"""
    querystring = BaseContentSearchQuery()


class ContentGetterResult(BaseResponseSchema):
    """Content getter result"""
    content = SharedContentInfo(description="Shared content properties")


class ContentGetterResponse(MappingSchema):
    """Content getter response"""
    body = ContentGetterResult()
