#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from pyams_content.features.json import JSONBaseConverter
from pyams_content.features.json.interfaces import IJSONConverter
from pyams_content.shared.common.api.interfaces import REST_CONTENT_PUBLIC_GETTER_ROUTE_DEFAULT, \
    REST_CONTENT_PUBLIC_GETTER_ROUTE_SETTING
from pyams_content.shared.common.interfaces import IWfSharedContent
from pyams_content.shared.common.interfaces.types import IWfTypedSharedContent
from pyams_sequence.interfaces import ISequentialIdInfo
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.url import canonical_url


@adapter_config(required=(IWfSharedContent, IPyAMSLayer),
                provides=IJSONConverter)
class JSONSharedContentConverter(JSONBaseConverter):
    """Default shared content JSON converter"""

    getter_route_setting = REST_CONTENT_PUBLIC_GETTER_ROUTE_SETTING
    getter_route_default = REST_CONTENT_PUBLIC_GETTER_ROUTE_DEFAULT

    def convert_content(self, params):
        """Base context converter"""
        result = super().convert_content(params)
        context = self.context
        lang = params.get('lang')
        sequence = ISequentialIdInfo(context)
        result['oid'] = sequence.hex_oid
        result['base_oid'] = sequence.get_base_oid().strip()
        getter_route = self.request.registry.settings.get(self.getter_route_setting, self.getter_route_default)
        result['api_url'] = getter_route.format(content_type=self.context.content_type,
                                                oid=sequence.hex_oid)
        self.get_i18n_attribute(context, 'title', lang, result)
        if context.handle_short_name:
            self.get_i18n_attribute(context, 'short_name', lang, result)
        if context.handle_content_url:
            self.get_attribute(context, 'content_url', result)
            result['public_url'] = canonical_url(context, self.request)
        if context.handle_header:
            self.get_i18n_attribute(context, 'header', lang, result)
        if context.handle_description:
            self.get_i18n_attribute(context, 'description', lang, result)
        return result


@adapter_config(required=(IWfTypedSharedContent, IPyAMSLayer),
                provides=IJSONConverter)
class JSONTypedSharedContentConverter(JSONSharedContentConverter):
    """JSON typed shared content converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        data_type = self.context.get_data_type()
        if data_type is not None:
            result['data_type'] = data_type.__name__
        return result
