#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import locale
import logging

from hypatia.interfaces import ICatalog
from persistent import Persistent
from pyramid.events import subscriber
from pyramid.interfaces import IWSGIApplicationCreatedEvent
from pyramid.settings import asbool
from pyramid.threadlocal import get_current_registry
from zope.container.contained import Contained
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import implementer
from zope.intid.interfaces import IIntIds
from zope.lifecycleevent import ObjectModifiedEvent
from zope.lifecycleevent.interfaces import IObjectAddedEvent, IObjectModifiedEvent, IObjectRemovedEvent
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from pyams_catalog.utils import reindex_object
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.checker.interfaces import ERROR_VALUE, IContentChecker, \
    MISSING_LANG_VALUE, MISSING_VALUE
from pyams_content.features.json.interfaces import IJSONConverter
from pyams_content.features.review.interfaces import IReviewComments
from pyams_content.interfaces import CONTRIBUTOR_ROLE, GUEST_ROLE, IBaseContentInfo, \
    MANAGER_ROLE, OWNER_ROLE, READER_ROLE
from pyams_content.shared.common.interfaces import CONTENT_TYPES_VOCABULARY, IBaseSharedTool, \
    ISharedContent, ISharedSite, IWfSharedContent, IWfSharedContentCreator, IWfSharedContentFactory, \
    IWfSharedContentRoles, IWfSharedContentUpdater, SHARED_CONTENT_TYPES_VOCABULARY
from pyams_content.shared.common.interfaces.zmi import IDashboardTypeColumnValue
from pyams_i18n.content import I18nManagerMixin
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_portal.interfaces import DESIGNER_ROLE
from pyams_security.interfaces import IDefaultProtectionPolicy
from pyams_security.principal import UnknownPrincipal
from pyams_security.property import RolePrincipalsFieldProperty
from pyams_security.security import ProtectedObject
from pyams_security.utility import get_principal
from pyams_sequence.interfaces import ISequentialIdInfo, ISequentialIdTarget
from pyams_utils.adapter import ContextAdapter, adapter_config, get_adapter_weight
from pyams_utils.date import format_datetime
from pyams_utils.interfaces import VIEW_PERMISSION
from pyams_utils.property import classproperty, classproperty_support
from pyams_utils.registry import get_global_registry, get_utilities_for, get_utility, query_utility
from pyams_utils.request import check_request, query_request
from pyams_utils.timezone import tztime
from pyams_utils.traversing import get_parent
from pyams_utils.url import generate_url
from pyams_utils.vocabulary import vocabulary_config
from pyams_utils.zodb import volatile_property
from pyams_workflow.interfaces import IObjectClonedEvent, IWorkflow, IWorkflowPublicationSupport, IWorkflowState, \
    IWorkflowVersions

__docformat__ = 'restructuredtext'

from pyams_content import _


LOGGER = logging.getLogger('PyAMS (content)')


@vocabulary_config(name='PyAMS shared sites')
class SharedSiteVocabulary(SimpleVocabulary):
    """Shared sites vocabulary"""

    interface = ISharedSite

    def __init__(self, context):
        request = query_request()
        terms = [
            SimpleTerm(v, title=II18n(t).query_attribute('title', request=request))
            for v, t in get_utilities_for(self.interface)
        ]
        super().__init__(sorted(terms, key=lambda x: locale.strxfrm(x.title)))


#
# Content types management
#

CONTENT_TYPES = {}
"""Dictionnary of all registered content types"""

IGNORED_CONTENT_TYPES = {}
"""Dictionnary of all registered *ignored* content types; these content types match custom
tools which are handled as shared contents (like forms, images maps or views) but which are
not "real" contents."""


def register_content_type(content, shared_content=True):
    """Register a new content type"""
    CONTENT_TYPES[content.content_type] = content
    if not shared_content:
        IGNORED_CONTENT_TYPES[content.content_type] = content


@subscriber(IWSGIApplicationCreatedEvent)
def handle_content_types(event):
    """Check for content types to un-register"""
    registry = get_current_registry()
    for key, content in CONTENT_TYPES.copy().items():
        if not asbool(registry.settings.get('pyams_content.register.{0}'.format(key), True)):
            del CONTENT_TYPES[key]
            if key in IGNORED_CONTENT_TYPES:
                del IGNORED_CONTENT_TYPES[key]


@vocabulary_config(name=CONTENT_TYPES_VOCABULARY)
class ContentTypesVocabulary(SimpleVocabulary):
    """Content types vocabulary"""

    def __init__(self, context):
        request = check_request()
        translate = request.localizer.translate
        terms = sorted([
            SimpleTerm(content.content_type,
                       title=translate(content.content_name))
            for content in CONTENT_TYPES.values()
        ], key=lambda x: locale.strxfrm(x.title))
        super().__init__(terms)


@vocabulary_config(name=SHARED_CONTENT_TYPES_VOCABULARY)
class SharedContentTypesVocabulary(SimpleVocabulary):
    """Shared content types vocabulary"""

    def __init__(self, context):
        request = check_request()
        translate = request.localizer.translate
        terms = sorted([
            SimpleTerm(content.content_type,
                       title=translate(content.content_name))
            for content in CONTENT_TYPES.values()
            if content.content_type not in IGNORED_CONTENT_TYPES
        ], key=lambda x: locale.strxfrm(x.title))
        super().__init__(terms)


#
# Workflow shared content class and adapters
#

@implementer(IDefaultProtectionPolicy, IWfSharedContent, IWfSharedContentRoles,
             IWorkflowPublicationSupport)
class WfSharedContent(ProtectedObject, Persistent, Contained, I18nManagerMixin):
    """Shared data content class"""

    __roles__ = (OWNER_ROLE, MANAGER_ROLE, CONTRIBUTOR_ROLE, DESIGNER_ROLE, READER_ROLE,
                 GUEST_ROLE)
    roles_interface = IWfSharedContentRoles

    owner = RolePrincipalsFieldProperty(IWfSharedContentRoles['owner'])
    managers = RolePrincipalsFieldProperty(IWfSharedContentRoles['managers'])
    contributors = RolePrincipalsFieldProperty(IWfSharedContentRoles['contributors'])
    designers = RolePrincipalsFieldProperty(IWfSharedContentRoles['designers'])
    readers = RolePrincipalsFieldProperty(IWfSharedContentRoles['readers'])
    guests = RolePrincipalsFieldProperty(IWfSharedContentRoles['guests'])

    content_type = None
    content_name = None

    handle_short_name = False
    handle_content_url = True
    handle_header = True
    handle_description = True

    title = FieldProperty(IWfSharedContent['title'])
    short_name = FieldProperty(IWfSharedContent['short_name'])
    content_url = FieldProperty(IWfSharedContent['content_url'])
    creator = FieldProperty(IWfSharedContent['creator'])
    modifiers = FieldProperty(IWfSharedContent['modifiers'])
    last_modifier = FieldProperty(IWfSharedContent['last_modifier'])
    header = FieldProperty(IWfSharedContent['header'])
    description = FieldProperty(IWfSharedContent['description'])
    keywords = FieldProperty(IWfSharedContent['keywords'])
    notepad = FieldProperty(IWfSharedContent['notepad'])

    @property
    def first_owner(self):
        versions = IWorkflowVersions(self, None)
        if versions is not None:
            return versions.get_version(1).creator

    @property
    def creation_label(self):
        request = check_request()
        translate = request.localizer.translate
        return translate(_('{date} by {principal}')).format(
            date=format_datetime(tztime(IBaseContentInfo(self).created_date)),
            principal=get_principal(request, self.creator).title)

    @property
    def last_update_label(self):
        request = check_request()
        translate = request.localizer.translate
        return translate(_('{date} by {principal}')).format(
            date=format_datetime(tztime(IBaseContentInfo(self).modified_date)),
            principal=get_principal(request, self.last_modifier).title)

    def update_from_json(self, params, request=None):
        """Update content from JSON data"""
        result = None
        registry = get_global_registry()
        # check workflow update first
        wf_updater = registry.queryAdapter(self, IWfSharedContentUpdater, name='workflow')
        if wf_updater is not None:
            version = wf_updater.update_content(params, request)
            if version is not None:
                result = version
        # check other updaters
        if result is None:
            result = self
        wf_state = IWorkflowState(result)
        if wf_state.state in IWorkflow(result).update_states:
            for name, adapter in sorted(request.registry.getAdapters((result,),
                                                                     IWfSharedContentUpdater),
                                        key=get_adapter_weight):
                if name == 'workflow':
                    continue
                adapter.update_content(params, request)
        if params:
            registry = get_global_registry()
            registry.notify(ObjectModifiedEvent(result))
        return result

    def to_json(self, params=None):
        """Convert content to JSON"""
        request = check_request()
        converter = request.registry.queryMultiAdapter((self, request), IJSONConverter)
        return converter.to_json(params) if converter is not None else {}


@adapter_config(required=IWfSharedContent,
                provides=IWfSharedContentCreator)
class WfSharedContentCreator(ContextAdapter):
    """Default shared content creator"""

    weight = 0

    def update_content(self, params, request=None):
        """Update new content version with given params"""
        if request is None:
            request = check_request()
        negotiator = get_utility(INegotiator)
        lang = negotiator.server_language
        title = params.get('title')
        self.context.title = title if isinstance(title, dict) else {lang: title}
        if self.context.handle_short_name:
            short_name = params.get('short_name')
            if short_name:
                self.context.short_name = short_name if isinstance(short_name, dict) else {lang: short_name}
            else:
                self.context.short_name = self.context.title.copy()
        if self.context.handle_content_url:
            content_url = params.get('content_url')
            if content_url:
                self.context.content_url = generate_url(content_url)
            else:
                self.context.content_url = generate_url(self.context.title.get(lang, ''))
        if self.context.handle_header:
            header = params.get('header')
            if header:
                self.context.header = header if isinstance(header, dict) else {lang: header}
        if self.context.handle_description:
            description = params.get('description')
            if description:
                self.context.description = description if isinstance(description, dict) else {lang: description}
        self.context.creator = request.principal.id
        self.context.owner = request.principal.id


@adapter_config(required=IWfSharedContent,
                provides=IWfSharedContentUpdater)
class WfSharedContentUpdater(ContextAdapter):
    """Default shared content updater"""

    def update_content(self, params, request=None):
        """Update existing content version with given params"""
        negotiator = get_utility(INegotiator)
        lang = negotiator.server_language
        if 'title' in params:
            title = params['title']
            self.context.title = title if isinstance(title, dict) else {lang: title}
        if self.context.handle_short_name and ('short_name' in params):
            short_name = params['short_name']
            self.context.short_name = short_name if isinstance(short_name, dict) else {lang: short_name}
        if self.context.handle_content_url and ('content_url' in params):
            content_url = params['content_url']
            self.context.content_url = generate_url(content_url)
        if self.context.handle_header and ('header' in params):
            header = params['header']
            self.context.header = header if isinstance(header, dict) else {lang: header}
        if self.context.handle_description and ('description' in params):
            description = params['description']
            self.context.description = description if isinstance(description, dict) else {lang: description}


@subscriber(IObjectModifiedEvent, context_selector=IWfSharedContent)
def handle_modified_shared_content(event):
    """Define content's modifiers when content is modified"""
    request = query_request()
    if request is not None:
        try:
            principal_id = request.principal.id
        except AttributeError:
            pass
        else:
            if principal_id != UnknownPrincipal.id:
                content = event.object
                modifiers = content.modifiers or set()
                if principal_id not in modifiers:
                    modifiers.add(principal_id)
                    content.modifiers = modifiers
                    catalog = query_utility(ICatalog)
                    intids = query_utility(IIntIds)
                    catalog['modifiers'].reindex_doc(intids.register(content), content)
                content.last_modifier = principal_id


@subscriber(IObjectAddedEvent)
@subscriber(IObjectModifiedEvent)
@subscriber(IObjectRemovedEvent)
def handle_modified_inner_content(event):
    """Handle modified shared object inner content

    This generic subscriber is used to update index on any content modification.
    """
    source = event.object
    if IWfSharedContent.providedBy(source):
        return
    content = get_parent(event.object, IWfSharedContent, allow_context=False)
    if content is None:
        return
    handle_modified_shared_content(ObjectModifiedEvent(content))
    reindex_object(content)


@subscriber(IObjectClonedEvent, context_selector=IWfSharedContent)
def handle_cloned_shared_content(event):
    """Handle a cloned object when a new version is created

    Current principal is set as version creator, and is added to version
    contributors if he is not the original content's owner
    """
    request = query_request()
    principal_id = request.principal.id
    content = event.object
    content.creator = principal_id
    if principal_id not in content.owner:
        # creation of new versions doesn't change owner
        # but new creators are added to contributors list
        contributors = content.contributors or set()
        contributors.add(principal_id)
        content.contributors = contributors
    # reset modifiers
    content.modifiers = set()
    # clear review comments
    comments = IReviewComments(content, None)
    if comments is not None:
        comments.clear()


@adapter_config(context=IWfSharedContent,
                provides=ISequentialIdInfo)
def wf_shared_content_sequence_adapter(context):
    """Shared content sequence adapter"""
    parent = get_parent(context, ISharedContent)
    if parent is not None:
        return ISequentialIdInfo(parent)


@adapter_config(context=IWfSharedContent,
                provides=IBaseContentInfo)
class WfSharedContentInfoAdapter(ContextAdapter):
    """Shared content base info adapter"""

    @property
    def created_date(self):
        return IZopeDublinCore(self.context).created

    @property
    def modified_date(self):
        return IZopeDublinCore(self.context).modified


@adapter_config(name='properties',
                context=IWfSharedContent,
                provides=IContentChecker)
class WfSharedContentChecker(BaseContentChecker):
    """Default shared content checker"""

    label = _("Properties")
    required_attributes = ('title', 'short_name', 'description')

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        missing_value = translate(MISSING_VALUE)
        missing_lang_value = translate(MISSING_LANG_VALUE)
        i18n = II18n(self.context)
        langs = II18nManager(self.context).get_languages()
        for attr in self.required_attributes:
            for lang in langs:
                value = i18n.get_attribute(attr, lang, request)
                if not value:
                    if len(langs) == 1:
                        output.append(
                            missing_value.format(field=translate(IWfSharedContent[attr].title)))
                    else:
                        output.append(missing_lang_value.format(
                            field=translate(IWfSharedContent[attr].title),
                            lang=lang))
                else:
                    length = len(value)
                    if (attr == 'title') and (length < 40 or length > 66):
                        output.append(translate(ERROR_VALUE).format(
                            field=translate(IWfSharedContent[attr].title),
                            message=translate(
                                _("title length should be between 40 and 66 characters ({length} "
                                  "actually)")).format(length=length)
                        ))
        if not self.context.keywords:
            output.append(
                missing_value.format(field=translate(IWfSharedContent['keywords'].title)))
        return output


#
# Main shared content class and adapters
#

@classproperty_support
@implementer(ISharedContent, ISequentialIdTarget)
class SharedContent(Persistent, Contained):
    """Workflow managed shared data"""

    view_permission = VIEW_PERMISSION

    sequence_name = ''  # use default sequence generator
    sequence_prefix = ''

    @classproperty
    def content_class(cls):
        return IWfSharedContentFactory(cls, None)

    @property
    def workflow_name(self):
        return get_parent(self, IBaseSharedTool).shared_content_workflow

    @volatile_property
    def visible_version(self):
        workflow = IWorkflow(self)
        versions = IWorkflowVersions(self).get_versions(workflow.visible_states, sort=True)
        if versions:
            return versions[-1]


@adapter_config(context=ISharedContent, provides=IBaseContentInfo)
class SharedContentInfoAdapter(ContextAdapter):
    """Shared content base info adapter"""

    @property
    def created_date(self):
        return IZopeDublinCore(self.context).created

    @property
    def modified_date(self):
        return IZopeDublinCore(self.context).modified
