#
# Copyright (c) 2008-2016 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library
from datetime import datetime

from pyramid.decorator import reify
from pyramid.path import DottedNameResolver
from pyramid.view import view_config
from zope.interface import Interface
from zope.lifecycleevent import ObjectCreatedEvent

from pyams_content import _
# import interfaces
from pyams_content.interfaces import CREATE_CONTENT_PERMISSION
from pyams_content.shared.blog.interfaces import IWfBlogPost, IBlogManager
# import packages
from pyams_content.shared.common.zmi import SharedContentAddForm, SharedContentAJAXAddForm
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces import IContentTitle
from pyams_skin.interfaces.viewlet import IMenuHeader, IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import adapter_config, ContextRequestViewAdapter, ContextRequestAdapter
from pyams_utils.registry import get_utility
from pyams_utils.traversing import get_parent
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url, generate_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_workflow.interfaces import IWorkflowVersions, IWorkflowInfo
from pyams_zmi.interfaces.menu import IContentManagementMenu
from pyams_zmi.layer import IAdminLayer


@adapter_config(context=(IWfBlogPost, IContentManagementMenu), provides=IMenuHeader)
class BlogPostContentMenuHeader(ContextRequestAdapter):
    """Blog post content menu header adapter"""

    header = _("This blog post")


@adapter_config(context=(IWfBlogPost, IPyAMSLayer, Interface), provides=IContentTitle)
class BlogPostTitleAdapter(ContextRequestViewAdapter):
    """Blog post title adapter"""

    @property
    def title(self):
        translate = self.request.localizer.translate
        return translate(_("Blog post « {title} »")).format(
            title=II18n(self.context).query_attribute('title', request=self.request))


@viewlet_config(name='add-shared-content.action', context=IBlogManager, layer=IAdminLayer, view=Interface,
                manager=IWidgetTitleViewletManager, permission=CREATE_CONTENT_PERMISSION, weight=1)
class BlogPostAddAction(ToolbarAction):
    """Blog post adding action"""

    label = _("Add blog post")
    url = 'add-shared-content.html'
    modal_target = True


@pagelet_config(name='add-shared-content.html', context=IBlogManager, layer=IPyAMSLayer,
                permission=CREATE_CONTENT_PERMISSION)
class BlogPostAddForm(SharedContentAddForm):
    """Blog post add form"""

    legend = _("Add blog post")
    content_url = None

    __target = None

    @reify
    def content_factory(self):
        registry = self.request.registry
        factory = registry.settings.get('pyams_content.config.blogpost_factory')
        if factory:
            factory = DottedNameResolver().resolve(factory)
        else:
            manager = get_parent(self.context, IBlogManager)
            factory = manager.topic_content_factory
        return factory

    def create(self, data):
        return self.content_factory.content_class()

    def update_content(self, content, data):
        data = data.get(self, data)
        # initialize content fields
        lang = get_utility(INegotiator).server_language
        content.creator = self.request.principal.id
        content.owner = self.request.principal.id
        content.title = data['title']
        content.short_name = content.title.copy()
        content.content_url = generate_url(content.title.get(lang, ''))
        content.notepad = data.get('notepad')
        # check blog folders
        now = datetime.utcnow()
        year, month = now.strftime('%Y:%m').split(':')
        year_folder = self.context.get(year)
        if year_folder is None:
            year_folder = self.context[year] = self.context.folder_factory()
        month_folder = year_folder.get(month)
        if month_folder is None:
            month_folder = year_folder[month] = self.context.folder_factory()
        # update languages
        languages = II18nManager(self.context).languages
        if languages:
            II18nManager(content).languages = languages.copy()
        # create content
        wf_parent = self.content_factory()
        self.request.registry.notify(ObjectCreatedEvent(wf_parent))
        # add content to blog folder
        title = II18n(content).query_attribute('title', request=self.request)
        month_folder[translate_string(title, spaces='-')] = wf_parent
        IWorkflowVersions(wf_parent).add_version(content, None)
        IWorkflowInfo(content).fire_transition('init', comment=content.notepad)
        self.__target = content

    def add(self, content):
        pass

    def nextURL(self):
        return absolute_url(self.__target, self.request, 'admin')


@view_config(name='add-shared-content.json', context=IBlogManager, request_type=IPyAMSLayer,
             permission=CREATE_CONTENT_PERMISSION, renderer='json', xhr=True)
class BlogPostAJAXAddForm(SharedContentAJAXAddForm, BlogPostAddForm):
    """Blog post add form, JSON renderer"""
