#
# Copyright (c) 2008-2016 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.events import subscriber
from zope.component.interfaces import ISite
from zope.container.folder import Folder
from zope.interface import implementer
from zope.lifecycleevent.interfaces import IObjectAddedEvent, IObjectRemovedEvent
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.illustration import IIllustrationTarget, ILinkIllustrationTarget
from pyams_content.component.paragraph.interfaces import IParagraphFactorySettings
from pyams_content.component.theme.interfaces import IThemesManagerTarget
from pyams_content.features.footer.interfaces import IFooterTarget
from pyams_content.features.header.interfaces import IHeaderTarget
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.reference.pictograms.interfaces import IPictogramManagerTarget
from pyams_content.root.interfaces import ISiteRoot
from pyams_content.shared.blog import BlogPost
from pyams_content.shared.blog.interfaces import IBlogManager, IBlogFolder, IBlogFolderFactory, IBlogManagerFactory
from pyams_content.shared.common.interfaces import ISharedContentFactory
from pyams_content.shared.common.manager import BaseSharedTool
from pyams_portal.interfaces import IPortalContext
from pyams_skin.skin import UserSkinnableContent
from pyams_utils.adapter import adapter_config
from pyams_utils.traversing import get_parent


@implementer(IBlogFolder)
class BlogFolder(Folder):
    """Blog folder class"""

    def is_deletable(self):
        for element in self.values():
            if not element.is_deletable():
                return False
        return True


@implementer(IBlogManager, IParagraphFactorySettings, IThemesManagerTarget,
             IPictogramManagerTarget, IIllustrationTarget, ILinkIllustrationTarget,
             IHeaderTarget, IFooterTarget, IPortalContext, IPreviewTarget)
class BlogManager(Folder, BaseSharedTool, UserSkinnableContent):
    """Nlog manager class"""

    description = FieldProperty(IBlogManager['description'])
    notepad = FieldProperty(IBlogManager['notepad'])

    allowed_paragraphs = FieldProperty(IParagraphFactorySettings['allowed_paragraphs'])
    auto_created_paragraphs = FieldProperty(IParagraphFactorySettings['auto_created_paragraphs'])

    sequence_name = ''  # use default sequence generator
    sequence_prefix = ''

    @property
    def folder_factory(self):
        return IBlogFolderFactory(self, BlogFolder)

    @property
    def topic_content_factory(self):
        return ISharedContentFactory(self, None)

    @property
    def topic_content_type(self):
        factory = self.topic_content_factory
        if factory is not None:
            return factory.content_class.content_type

    def is_deletable(self):
        for element in self.values():
            if not element.is_deletable():
                return False
        return True


@adapter_config(context=ISiteRoot, provides=IBlogManagerFactory)
def site_root_blog_manager_factory(context):
    return BlogManager


@adapter_config(context=IBlogManager, provides=IBlogFolderFactory)
def blog_manager_folder_factory(context):
    return BlogFolder


@adapter_config(context=IBlogManager, provides=ISharedContentFactory)
def blog_manager_topic_factory(context):
    return BlogPost


@subscriber(IObjectAddedEvent, context_selector=IBlogManager)
def handle_added_blog_manager(event):
    """Register blog manager when added"""
    site = get_parent(event.newParent, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, IBlogManager, name=event.object.__name__)


@subscriber(IObjectRemovedEvent, context_selector=IBlogManager)
def handle_deleted_blog_manager(event):
    """Un-register blog manager when deleted"""
    site = get_parent(event.oldParent, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.unregisterUtility(event.object, IBlogManager, name=event.object.__name__)
