#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

import json

from pyramid.decorator import reify
from pyramid.events import subscriber
from pyramid.view import view_config
from z3c.form.field import Fields
from z3c.form.interfaces import DISPLAY_MODE, IDataExtractedEvent
from z3c.table.interfaces import IColumn
from zope.interface import Invalid, alsoProvides
from zope.schema.vocabulary import getVocabularyRegistry

from pyams_content.interfaces import MANAGE_TOOL_PERMISSION
from pyams_content.reference.pictograms.zmi.widget import PictogramSelectFieldWidget
from pyams_content.shared.alert.interfaces import IAlertsManager, IAlertTypesManager, IAlertType, ALERT_TYPES_VOCABULARY
from pyams_content.shared.alert.type import AlertType
from pyams_content.zmi import pyams_content
from pyams_form.form import ajax_config, AJAXAddForm
from pyams_form.security import ProtectedFormObjectMixin
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.container import delete_container_element
from pyams_skin.interfaces.container import ITableElementName
from pyams_skin.interfaces.viewlet import IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import BaseTable, SorterColumn, NameColumn, TrashColumn
from pyams_skin.viewlet.menu import MenuDivider, MenuItem
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import adapter_config, ContextRequestAdapter
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import ContainerAdminView

from pyams_content import _


@viewlet_config(name='alert-types.divider',
                context=IAlertsManager, layer=IAdminLayer,
                manager=IPropertiesMenu, weight=599,
                permission=MANAGE_TOOL_PERMISSION)
class AlertTypesMenuDivider(MenuDivider):
    """Alert types menu divider"""


@viewlet_config(name='alert-types.menu',
                context=IAlertsManager, layer=IAdminLayer,
                manager=IPropertiesMenu, weight=600,
                permission=MANAGE_TOOL_PERMISSION)
class AlertTypesMenu(MenuItem):
    """Alert types menu"""

    label = _("Alerts types")
    icon_class = 'fa-folder-o'
    url = '#alert-types.html'


#
# Alert types manager views
#

class AlertTypesTable(ProtectedFormObjectMixin, BaseTable):
    """Alert types table"""

    prefix = 'types'

    hide_header = True
    sortOn = None

    @property
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover', 'table-tight']
        permission = self.permission
        if (not permission) or self.request.has_permission(permission, context=self.context):
            classes.append('table-dnd')
        return {'table': ' '.join(classes)}

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes['table'] = {
            'id': self.id,
            'data-ams-plugins': 'pyams_content',
            'data-ams-plugin-pyams_content-src': get_resource_path(pyams_content),
            'data-ams-location': absolute_url(IAlertTypesManager(self.context), self.request),
            'data-ams-tablednd-drag-handle': 'td.sorter',
            'data-ams-tablednd-drop-target': 'set-types-order.json'
        }
        return attributes

    @reify
    def values(self):
        return list(IAlertTypesManager(self.context).values())

    def render(self):
        if not self.values:
            translate = self.request.localizer.translate
            return translate(_("No currently defined alert type."))
        return super().render()


@adapter_config(name='sorter',
                required=(IAlertsManager, IAdminLayer, AlertTypesTable),
                provides=IColumn)
class AlertTypesSorterColumn(ProtectedFormObjectMixin, SorterColumn):
    """Alert types sorter column"""


@view_config(name='set-types-order.json',
             context=IAlertTypesManager, request_type=IPyAMSLayer,
             permission=MANAGE_TOOL_PERMISSION, renderer='json', xhr=True)
def set_alert_types_order(request):
    """Update alert types order"""
    order = list(map(str, json.loads(request.params.get('names'))))
    request.context.updateOrder(order)
    return {'status': 'success'}


@adapter_config(name='name',
                required=(IAlertsManager, IPyAMSLayer, AlertTypesTable),
                provides=IColumn)
class AlertTypesNameColumn(NameColumn):
    """Alert types name column"""

    _header = _("Type label")


@adapter_config(name='trash',
                required=(IAlertsManager, IPyAMSLayer, AlertTypesTable),
                provides=IColumn)
class AlertTypesTrashColumn(TrashColumn):
    """Alert types trash column"""

    permission = MANAGE_TOOL_PERMISSION


@view_config(name='delete-element.json',
             context=IAlertTypesManager, request_type=IPyAMSLayer,
             permission=MANAGE_TOOL_PERMISSION, renderer='json', xhr=True)
def delete_alert_type(request):
    """Data alert delete view"""
    return delete_container_element(request, ignore_permission=True)


@pagelet_config(name='alert-types.html',
                context=IAlertsManager, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
class AlertTypesView(ContainerAdminView):
    """Alerts types view"""

    title = _("Alert types")
    table_class = AlertTypesTable


#
# Alert types forms
#

@adapter_config(required=(IAlertType, IPyAMSLayer), 
                provides=ITableElementName)
class AlertTypeElementNameAdapter(ContextRequestAdapter):
    """Alert type name adapter"""

    @property
    def name(self):
        return II18n(self.context).query_attribute('label', request=self.request)


@viewlet_config(name='add-alert-type.action', 
                context=IAlertsManager, layer=IAdminLayer,
                view=AlertTypesView, manager=IWidgetTitleViewletManager, weight=1,
                permission=MANAGE_TOOL_PERMISSION)
class AlertTypeAddAction(ToolbarAction):
    """Alert type adding action"""

    label = _("Add alert type")
    label_css_class = 'fa fa-fw fa-plus'
    url = 'add-alert-type.html'
    modal_target = True


@pagelet_config(name='add-alert-type.html',
                context=IAlertsManager, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='add-alert-type.json',
             context=IAlertsManager, layer=IPyAMSLayer,
             base=AJAXAddForm)
class AlertTypeAddForm(AdminDialogAddForm):
    """Alert type add form"""
    
    legend = _("Add new alert type")
    icon_css_class = 'fa fa-fw fa-folder-o'
    label_css_class = 'control-label col-md-4'
    input_css_class = 'col-md-8'
    
    fields = Fields(IAlertType).omit('__parent__', '__name__')
    fields['pictogram'].widgetFactory = PictogramSelectFieldWidget

    edit_permission = MANAGE_TOOL_PERMISSION

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        color = self.widgets.get('color')
        if color is not None:
            color.object_data = {
                'ams-colorpicker-position': 'top left'
            }
            alsoProvides(color, IObjectData)

    def create(self, data):
        return AlertType()
    
    def add(self, object):
        name = translate_string(object.name, spaces='-')
        IAlertTypesManager(self.context)[name] = object
        
    def nextURL(self):
        return '#alert-types.html'
    
    
@subscriber(IDataExtractedEvent, form_selector=AlertTypeAddForm)
def handle_datatype_add_form_data_extraction(event):
    """Check new alert type for existing name"""
    name = event.data.get('name')
    typename = translate_string(name, spaces='-')
    vocabulary = getVocabularyRegistry().get(event.form.context, ALERT_TYPES_VOCABULARY)
    try:
        vocabulary.getTerm(typename)
        event.form.widgets.errors += (Invalid(_("Specified alert type name is already used!")),)
    except LookupError:
        pass


@pagelet_config(name='properties.html', 
                context=IAlertType, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='properties.json',
             context=IAlertType, layer=IPyAMSLayer)
class AlertTypeEditForm(AdminDialogEditForm):
    """Alert type edit form"""

    prefix = 'alert_type_properties.'

    legend = _("Alert type properties")
    icon_css_class = 'fa fa-fw fa-folder-o'
    label_css_class = 'control-label col-md-4'
    input_css_class = 'col-md-8'

    fields = Fields(IAlertType).omit('__parent__', '__name__')
    fields['pictogram'].widgetFactory = PictogramSelectFieldWidget

    edit_permission = MANAGE_TOOL_PERMISSION

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        name = self.widgets.get('name')
        if name is not None:
            name.mode = DISPLAY_MODE
        color = self.widgets.get('color')
        if color is not None:
            color.object_data = {
                'ams-colorpicker-position': 'top left'
            }
            alsoProvides(color, IObjectData)
