#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from cornice import Service
from pyramid.httpexceptions import HTTPOk, HTTPUnauthorized, HTTPServiceUnavailable
from pyramid.security import Authenticated

from pyams_content.features.json import JSONBaseConverter, to_json
from pyams_content.features.json.interfaces import IJSONConverter
from pyams_content.shared.alert import IAlertsManager, IAlertTypesManager
from pyams_content.shared.alert.api.interfaces import REST_ALERT_TYPE_GETTER_ROUTE
from pyams_content.shared.alert.api.schema import AlertTypesGetterResponse
from pyams_content.shared.alert.interfaces import IWfAlert
from pyams_security.rest import check_cors_origin, set_cors_headers
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.registry import query_utility
from pyams_utils.request import check_request
from pyams_utils.rest import rest_responses, http_error, http_response
from pyams_utils.url import canonical_url


#
# Alert types service
#

alert_types_getter_service = Service(name=REST_ALERT_TYPE_GETTER_ROUTE,
                                     pyramid_route=REST_ALERT_TYPE_GETTER_ROUTE,
                                     description="Alert types getter service")


@alert_types_getter_service.options(validators=(check_cors_origin, set_cors_headers))
def alert_types_getter_service_options(request):
    return ''


alert_types_getter_responses = rest_responses.copy()
alert_types_getter_responses[HTTPOk.code] = AlertTypesGetterResponse(description="Alert types getter results")


@alert_types_getter_service.get(validators=(check_cors_origin, set_cors_headers),
                                response_schemas=alert_types_getter_responses)
def alert_types_getter_service_get(request):
    """Alert types service getter"""
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    alerts_manager = query_utility(IAlertsManager)
    if alerts_manager is None:
        return http_error(request, HTTPServiceUnavailable)
    alerts_types = IAlertTypesManager(alerts_manager, None)
    if alerts_types is None:
        return http_error(request, HTTPServiceUnavailable)
    return http_response(request, HTTPOk, {
        'alert_types': [
            to_json(alert_type, request)
            for alert_type in alerts_types.values()
        ]
    })


#
# JSON alert specificities converter
#

@adapter_config(name='alert',
                required=(IWfAlert, IPyAMSLayer),
                provides=IJSONConverter)
class JSONAlertConverter(JSONBaseConverter):
    """JSON alert converter"""

    is_inner = True

    @property
    def conversion_target(self):
        return None

    def convert_content(self, params):
        result = super().convert_content(params)
        context = self.context
        self.get_attribute(context, 'alert_type', result)
        self.get_i18n_attribute(context, 'body', params.get('lang'), result)
        if context.reference:
            target = context.target
            if target is not None:
                request = check_request()
                result['href'] = canonical_url(target, request)
        elif context.external_url:
            result['href'] = context.external_url
        self.get_attribute(context, 'maximum_interval', result)
        return result
