#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.decorator import reify
from pyramid.view import view_config
from z3c.form import field
from z3c.table.interfaces import IColumn, IValues
from zope.interface import implementer

from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION
from pyams_content.reference.interfaces import IReferenceTable
from pyams_form.form import ajax_config
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.container import ContainerView, delete_container_element
from pyams_skin.interfaces import IInnerPage, IPageHeader
from pyams_skin.interfaces.viewlet import IBreadcrumbItem, IMenuHeader
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.page import DefaultPageHeaderAdapter
from pyams_skin.table import BaseTable, TrashColumn
from pyams_skin.viewlet.breadcrumb import BreadcrumbAdminLayerItem
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.adapter import ContextRequestAdapter, ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.url import absolute_url
from pyams_viewlet.manager import viewletmanager_config
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu, ISiteManagementMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import AdminView

__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Table properties
#

@adapter_config(context=(IReferenceTable, IAdminLayer), provides=IBreadcrumbItem)
class ReferenceTableBreadcrumbAdapter(BreadcrumbAdminLayerItem):
    """References table breadcrumb adapter"""

    @property
    def label(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    css_class = 'strong'


@viewlet_config(name='contents.menu', context=IReferenceTable, layer=IAdminLayer,
                manager=ISiteManagementMenu, permission=VIEW_SYSTEM_PERMISSION, weight=10)
@viewletmanager_config(name='contents.menu', layer=IAdminLayer, provides=IPropertiesMenu)
@implementer(IPropertiesMenu)
class ReferenceTableContentsMenu(MenuItem):
    """References table contents menu"""

    label = _("Contents")
    icon_class = 'fa-table'
    url = '#contents.html'


class ReferenceTableContentsTable(BaseTable):
    """References table contents table"""

    prefix = 'references'

    title = _("Table contents")

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-location': absolute_url(self.context, self.request)
        })
        return attributes


@adapter_config(context=(IReferenceTable, IPyAMSLayer, ReferenceTableContentsTable), provides=IValues)
class ReferenceTableContentsValues(ContextRequestViewAdapter):
    """Reference table contents values"""

    @property
    def values(self):
        return self.context.values()


@adapter_config(name='trash', context=(IReferenceTable, IPyAMSLayer, ReferenceTableContentsTable), provides=IColumn)
class ReferenceTableTrashColumn(TrashColumn):
    """Reference table trash column"""

    permission = MANAGE_SITE_ROOT_PERMISSION


@view_config(name='delete-element.json', context=IReferenceTable, request_type=IPyAMSLayer,
             permission=MANAGE_SITE_ROOT_PERMISSION, renderer='json', xhr=True)
def delete_reference(request):
    """Delete selected reference item"""
    return delete_container_element(request, ignore_permission=True)


@pagelet_config(name='contents.html', context=IReferenceTable, layer=IPyAMSLayer, permission=VIEW_SYSTEM_PERMISSION)
@implementer(IInnerPage)
class ReferenceTableContentsView(AdminView, ContainerView):
    """Reference table contents view"""

    table_class = ReferenceTableContentsTable


@adapter_config(context=(IReferenceTable, IAdminLayer, ReferenceTableContentsView), provides=IPageHeader)
class ReferenceTableHeaderAdapter(DefaultPageHeaderAdapter):
    """References table header adapter"""

    title = _("References tables")

    @property
    def subtitle(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    icon_class = 'fa fa-fw fa-table'


@viewlet_config(name='properties.menu', context=IReferenceTable, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION, weight=10)
class ReferenceTablePropertiesMenu(MenuItem):
    """Reference table properties menu"""

    label = _("Properties...")
    icon_class = 'fa-edit'

    url = 'properties.html'
    modal_target = True


@pagelet_config(name='properties.html', context=IReferenceTable, layer=IPyAMSLayer, permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json', context=IReferenceTable, layer=IPyAMSLayer)
class ReferenceTablePropertiesEditForm(AdminDialogEditForm):
    """Reference table properties edit form"""

    prefix = 'table_properties.'

    legend = _("Edit table properties")

    fields = field.Fields(IReferenceTable).omit('__parent__', '__name__')
    edit_permission = MANAGE_SITE_ROOT_PERMISSION


@adapter_config(context=(IReferenceTable, ISiteManagementMenu), provides=IMenuHeader)
class ReferenceTableSiteManagementMenuHeader(ContextRequestAdapter):
    """Reference table site management menu header adapter"""

    header = _("Table management")
