#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

import json
import locale

from z3c.form import button, field
from z3c.form.util import changedField
from zope.interface import Interface, implementer

from pyams_content.interfaces import MANAGE_TOOL_PERMISSION
from pyams_content.reference.pictograms.interfaces import IPictogramManager, IPictogramManagerTarget, IPictogramTable
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IUncheckedEditFormButtons, IWidgetForm, IWidgetsSuffixViewletsManager
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces import IInnerPage
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.menu import MenuItem
from pyams_template.template import template_config
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.registry import query_utility
from pyams_utils.traversing import get_parent
from pyams_viewlet.viewlet import Viewlet, viewlet_config
from pyams_zmi.form import AdminEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer

from pyams_content import _


@viewlet_config(name='pictograms-selection.menu', context=IPictogramManagerTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=MANAGE_TOOL_PERMISSION, weight=760)
class PictogramManagerMenu(MenuItem):
    """Pictogram manager menu"""

    label = _("Pictograms selection...")
    icon_class = 'fa-linode'
    url = '#pictograms-selection.html'


@pagelet_config(name='pictograms-selection.html', context=IPictogramManagerTarget, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='pictograms-selection.json', context=IPictogramManagerTarget, layer=IPyAMSLayer)
@implementer(IWidgetForm, IInnerPage, IObjectData)
class PictogramManagerEditForm(AdminEditForm):
    """Pictogram manager selection form"""

    legend = _("Selected pictograms")

    fields = field.Fields(Interface)
    buttons = button.Buttons(IUncheckedEditFormButtons)

    object_data = {'ams-form-data-init-callback': 'PyAMS_content.pictograms.initManagerSelection'}

    edit_permission = MANAGE_TOOL_PERMISSION

    def update_content(self, content, data):
        changes = {}
        manager = IPictogramManager(self.context)
        field = IPictogramManager['selected_pictograms']
        selected = json.loads(self.request.params.get('selected', '[]'))
        if changedField(field, selected, context=manager):
            manager.selected_pictograms = selected
            changes.setdefault(IPictogramManager, []).append(field.__name__)
        return changes


@viewlet_config(name='pictogram-selection.subform', context=IPictogramManagerTarget, layer=IAdminLayer,
                view=PictogramManagerEditForm, manager=IWidgetsSuffixViewletsManager, permission=MANAGE_TOOL_PERMISSION)
@template_config(template='templates/manager-selection.pt', layer=IAdminLayer)
class PictogramManagerSelectionView(Viewlet):
    """Pictogram manager selection viewlet"""

    pictogram_table = None
    pictogram_manager = None

    def update(self):
        super(PictogramManagerSelectionView, self).update()
        self.pictogram_table = query_utility(IPictogramTable)
        target = get_parent(self.context, IPictogramManagerTarget)
        if target is not None:
            self.pictogram_manager = IPictogramManager(target)

    @property
    def sorted_pictograms(self):
        yield from sorted(self.pictogram_table.values(),
                          key=lambda x: locale.strxfrm((II18n(x).query_attribute('title',
                                                                                 request=self.request) or '').lower()))

    @property
    def available_pictograms(self):
        manager = self.pictogram_manager
        if manager is not None:
            for pictogram in self.sorted_pictograms:
                if pictogram.__name__ not in (manager.selected_pictograms or ()):
                    yield pictogram

    @property
    def selected_pictograms(self):
        manager = self.pictogram_manager
        if manager is not None:
            for name in (manager.selected_pictograms or ()):
                pictogram = self.pictogram_table.get(name)
                if pictogram is not None:
                    yield pictogram
