#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from uuid import uuid4

from pyramid.decorator import reify
from pyramid.httpexceptions import HTTPNotFound
from pyramid.renderers import render, render_to_response
from pyramid.response import Response
from pyramid.view import view_config
from z3c.form import field
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn

from pyams_content.interfaces import IBaseContent, MANAGE_SITE_ROOT_PERMISSION
from pyams_content.reference.pictograms import Pictogram
from pyams_content.reference.pictograms.interfaces import IPictogram, IPictogramTable
from pyams_content.reference.zmi.table import ReferenceTableContentsTable, \
    ReferenceTableContentsView
from pyams_file.image import render_image
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_i18n.column import I18nAttrColumn
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_table_row_refresh_event
from pyams_skin.interfaces.viewlet import IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import I18nColumn
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.traversing import get_parent
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from pyams_zmi.layer import IAdminLayer

__docformat__ = 'restructuredtext'

from pyams_content import _


@viewlet_config(name='add-pictogram.action', context=IPictogramTable, layer=IPyAMSLayer,
                manager=IWidgetTitleViewletManager, view=ReferenceTableContentsTable,
                permission=MANAGE_SITE_ROOT_PERMISSION)
class PictogramAddAction(ToolbarAction):
    """Pictogram add action"""

    label = _("Add pictogram")

    url = 'add-pictogram.html'
    modal_target = True


@pagelet_config(name='add-pictogram.html', context=IPictogramTable, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='add-pictogram.json', context=IPictogramTable, layer=IPyAMSLayer,
             base=AJAXAddForm)
class PictogramAddForm(AdminDialogAddForm):
    """Pictogram add form"""

    legend = _("Add new pictogram")
    dialog_class = 'modal-large'

    fields = field.Fields(IPictogram).omit('__parent__', '__name__')
    edit_permission = MANAGE_SITE_ROOT_PERMISSION

    def create(self, data):
        return Pictogram()

    def add(self, object):
        name = str(uuid4())
        self.context[name] = object

    def nextURL(self):
        return 'contents.html'


@pagelet_config(name='properties.html', context=IPictogram, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json', context=IPictogram, layer=IPyAMSLayer)
class PictogramEditForm(AdminDialogEditForm):
    """Pictogram properties edit form"""

    prefix = 'pictogram_properties.'

    legend = _("Edit pictogram properties")
    dialog_class = 'modal-large'

    fields = field.Fields(IPictogram).omit('__parent__', '__name__')
    edit_permission = MANAGE_SITE_ROOT_PERMISSION

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if ('image' in changes.get(IPictogram, ())) or \
           ('title' in changes.get(IBaseContent, ())):
            parent = get_parent(self.context, IPictogramTable)
            output.setdefault('events', []).append(
                get_json_table_row_refresh_event(parent, self.request,
                                                 PictogramTableContentsTable, self.context))
        return output


#
# Pictogram table
#

class PictogramTableContentsTable(ReferenceTableContentsTable):
    """Pictograms table contents table"""

    prefix = 'pictograms_table'

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-datatable-sorting': '1,asc'
        })
        return attributes


@adapter_config(name='image',
                context=(IPictogramTable, IAdminLayer, PictogramTableContentsTable),
                provides=IColumn)
class PictogramTableImageColumn(GetAttrColumn):
    """Pictogram table image column"""

    header = ''
    weight = 1

    cssClasses = {'td': 'text-center width-50'}
    dt_sortable = 'false'

    def getValue(self, obj):
        image = II18n(obj).query_attribute('image', request=self.request)
        if image:
            return render_image(image, 32, 32, self.request, timestamp=True)
        return '--'


@adapter_config(name='name',
                context=(IPictogramTable, IAdminLayer, PictogramTableContentsTable),
                provides=IColumn)
class PictogramTableNameColumn(I18nColumn, I18nAttrColumn):
    """Pictogram table name column"""

    _header = _("Title")
    weight = 10
    attrName = 'title'


@pagelet_config(name='contents.html', context=IPictogramTable, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class PictogramTableContentsView(ReferenceTableContentsView):
    """Pictograms table contents view"""

    table_class = PictogramTableContentsTable


@view_config(name='get-pictogram-header.html',
             context=IPictogramTable, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION)
def get_pictogram_header_view(request):
    """View used to get thumbnail and alternate label associated with a given pictogram"""
    translate = request.localizer.translate
    name = request.params.get('value')
    if (not name) or (name == '--NOVALUE--'):
        return Response(translate(_("Default header: --")))
    pictogram = request.context.get(name)
    if pictogram is None:
        raise HTTPNotFound()
    return render_to_response('templates/pictogram-header.pt', {
        'context': pictogram
    }, request=request)


def get_pictogram_header(pictogram, request=None):
    """Get thumbnail and alternate label associated with a given pictogram"""
    return render('templates/pictogram-header.pt', {
        'context': pictogram
    }, request=request)
