#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.

import logging
from importlib import import_module

from hypatia.text.lexicon import Lexicon
from pyramid.path import DottedNameResolver
from zope.dublincore.interfaces import IZopeDublinCore
from zope.lifecycleevent import ObjectCreatedEvent

from pyams_catalog.index import DatetimeIndexWithInterface, DatetimeListIndexWithInterface, FieldIndexWithInterface, \
    KeywordIndexWithInterface
from pyams_catalog.interfaces import DATE_RESOLUTION, MINUTE_RESOLUTION
from pyams_catalog.nltk import NltkFullTextProcessor
from pyams_catalog.site import check_required_indexes
from pyams_content.component.calendar import ICalendarInfo
from pyams_content.component.theme import ICollectionsInfo, ITagsInfo, IThemesInfo
from pyams_content.interfaces import CONTRIBUTOR_ROLE, IBaseContent, MANAGER_ROLE, OWNER_ROLE, \
    PILOT_ROLE, WEBMASTER_ROLE
from pyams_content.reference import ReferencesManager
from pyams_content.reference.pictograms import PictogramTable
from pyams_content.root.interfaces import ISiteRootToolsConfiguration
from pyams_content.shared.alert.interfaces import IAlertsManagerFactory
from pyams_content.shared.calendar.interfaces import ICalendarManagerFactory
from pyams_content.shared.common.interfaces import IWfSharedContent
from pyams_content.shared.common.interfaces.types import IWfTypedSharedContent
from pyams_content.shared.common.manager import SharedToolContainer
from pyams_content.shared.file.interfaces import IFilesManagerFactory
from pyams_content.shared.form.interfaces import IFormsManagerFactory
from pyams_content.shared.imagemap.interfaces import IImageMapManagerFactory
from pyams_content.shared.logo.interfaces import ILogosManagerFactory
from pyams_content.shared.news.interfaces import INewsManagerFactory
from pyams_content.shared.resource.interfaces import IResourceManagerFactory
from pyams_content.shared.topic.interfaces import ITopicManagerFactory
from pyams_content.shared.view.interfaces import IViewsManagerFactory
from pyams_i18n.index import I18nTextIndexWithInterface
from pyams_security.index import PrincipalsRoleIndex
from pyams_thesaurus.index import ThesaurusTermsListFieldIndex
from pyams_utils.interfaces.site import ISiteGenerations
from pyams_utils.interfaces.traversing import IPathElements
from pyams_utils.registry import get_global_registry, utility_config
from pyams_utils.site import check_required_utilities
from pyams_workflow.interfaces import IWorkflowPublicationInfo, IWorkflowState


__docformat__ = 'restructuredtext'


logger = logging.getLogger("PyAMS (content)")


def get_fulltext_lexicon(language):
    return Lexicon(NltkFullTextProcessor(language=language))


RENAMED_CLASSES = {
    'pyams_content.component.association.menu MenusContainer':
        'pyams_content.features.menu MenusContainer',
    'pyams_content.component.association.menu Menu':
        'pyams_content.features.menu Menu',
    'pyams_content.component.paragraph.keynumber KeyNumber':
        'pyams_content.component.keynumber KeyNumber',
    'pyams_content.component.paragraph.keynumber KeyNumberContainer':
        'pyams_content.component.keynumber KeyNumberContainer',
    'pyams_content.component.paragraph.keynumber KeyNumberParagraph':
        'pyams_content.component.keynumber.paragraph KeyNumberParagraph',
    'pyams_content.shared.common.review ReviewComment':
        'pyams_content.features.review ReviewComment',
    'pyams_content.shared.common.review ReviewCommentsContainer':
        'pyams_content.features.review ReviewCommentsContainer',
    'pyams_content.shared.common.portlet.content SharedContentPortletSettings':
        'pyams_content.component.paragraph.portlet ParagraphContainerPortletSettings',
    'pyams_content.shared.site Topic':
        'pyams_content.shared.site SiteTopic',
    'pyams_content.shared.site WfTopic':
        'pyams_content.shared.site WfSiteTopic',
    'pyams_portal.portlets.content ContentPortletSettings':
        'pyams_content.portlet.content SharedContentPortletSettings',
    'pyams_content.portlet.content SharedContentPortletSettings':
        'pyams_content.shared.common.portlet.content SharedContentPortletSettings',
    'pyams_content.portlet.navigation SimpleNavigationPortletSettings':
        'pyams_content.features.menu.portlet.navigation.simple SimpleNavigationPortletSettings',
    'pyams_content.portlet.navigation.interfaces ISimpleNavigationMenu':
        'pyams_content.features.menu.portlet.navigation.interfaces.simple ISimpleNavigationMenu',
    'pyams_content.component.paragraph.interfaces.verbatim IVerbatimParagraph':
        'pyams_content.component.verbatim.interfaces IVerbatimParagraph',
    'pyams_content.component.paragraph.verbatim VerbatimParagraph':
        'pyams_content.component.verbatim.paragraph VerbatimParagraph',
    'pyams_content.component.paragraph.interfaces.map IMapParagraph':
        'pyams_content.component.map.interfaces IMapParagraph',
    'pyams_content.component.paragraph.map MapParagraph':
        'pyams_content.component.map.paragraph MapParagraph'
}

REQUIRED_UTILITIES = ()

REQUIRED_TABLES = (
    ('pictograms_table_name', 'pictograms', PictogramTable),
)

REQUIRED_TOOLS = [
    ('views', IViewsManagerFactory),
    ('logos', ILogosManagerFactory),
    ('files', IFilesManagerFactory),
    ('alerts', IAlertsManagerFactory),
    ('imagemaps', IImageMapManagerFactory),
    ('forms', IFormsManagerFactory),
    ('news', INewsManagerFactory),
    ('calendar', ICalendarManagerFactory),
    ('topics', ITopicManagerFactory),
    ('resources', IResourceManagerFactory)
]

REQUIRED_INDEXES = [
    ('content_type', FieldIndexWithInterface, {
        'interface': IBaseContent,
        'discriminator': 'content_type'
    }),
    ('data_type', FieldIndexWithInterface, {
        'interface': IWfTypedSharedContent,
        'discriminator': 'data_type'
    }),
    ('role:owner', PrincipalsRoleIndex, {
        'role_id': OWNER_ROLE
    }),
    ('role:pilot', PrincipalsRoleIndex, {
        'role_id': PILOT_ROLE
    }),
    ('role:manager', PrincipalsRoleIndex, {
        'role_id': MANAGER_ROLE
    }),
    ('role:contributor', PrincipalsRoleIndex, {
        'role_id': CONTRIBUTOR_ROLE
    }),
    ('role:webmaster', PrincipalsRoleIndex, {
        'role_id': WEBMASTER_ROLE}
     ),
    ('parents', KeywordIndexWithInterface, {
        'interface': IPathElements,
        'discriminator': 'parents'
    }),
    ('workflow_state', FieldIndexWithInterface, {
        'interface': IWorkflowState,
        'discriminator': 'state'
    }),
    ('workflow_principal', FieldIndexWithInterface, {
        'interface': IWorkflowState,
        'discriminator': 'state_principal'
    }),
    ('modifiers', KeywordIndexWithInterface, {
        'interface': IWfSharedContent,
        'discriminator': 'modifiers'
    }),
    ('created_date', DatetimeIndexWithInterface, {
        'interface': IZopeDublinCore,
        'discriminator': 'created',
        'resolution': DATE_RESOLUTION
    }),
    ('modified_date', DatetimeIndexWithInterface, {
        'interface': IZopeDublinCore,
        'discriminator': 'modified',
        'resolution': DATE_RESOLUTION
    }),
    ('publication_date', DatetimeIndexWithInterface, {
        'interface': IWorkflowPublicationInfo,
        'discriminator': 'publication_date',
        'resolution': MINUTE_RESOLUTION
    }),
    ('effective_date', DatetimeIndexWithInterface, {
        'interface': IWorkflowPublicationInfo,
        'discriminator': 'publication_effective_date',
        'resolution': MINUTE_RESOLUTION
    }),
    ('push_end_date', DatetimeIndexWithInterface, {
        'interface': IWorkflowPublicationInfo,
        'discriminator': 'push_end_date_index',
        'resolution': MINUTE_RESOLUTION
    }),
    ('expiration_date', DatetimeIndexWithInterface, {
        'interface': IWorkflowPublicationInfo,
        'discriminator': 'publication_expiration_date',
        'resolution': MINUTE_RESOLUTION
    }),
    ('first_publication_date', DatetimeIndexWithInterface, {
        'interface': IWorkflowPublicationInfo,
        'discriminator': 'first_publication_date',
        'resolution': MINUTE_RESOLUTION
    }),
    ('content_publication_date', DatetimeIndexWithInterface, {
        'interface': IWorkflowPublicationInfo,
        'discriminator': 'first_content_publication_date',
        'resolution': MINUTE_RESOLUTION
    }),
    ('visible_publication_date', DatetimeIndexWithInterface, {
        'interface': IWorkflowPublicationInfo,
        'discriminator': 'visible_publication_date',
        'resolution': MINUTE_RESOLUTION
    }),
    ('calendar_dates', DatetimeListIndexWithInterface, {
        'interface': ICalendarInfo,
        'discriminator': 'calendar',
        'resolution': DATE_RESOLUTION
    }),
    ('tags', ThesaurusTermsListFieldIndex, {
        'interface': ITagsInfo,
        'discriminator': 'tags',
        'include_parents': False,
        'include_synonyms': False
    }),
    ('themes', ThesaurusTermsListFieldIndex, {
        'interface': IThemesInfo,
        'discriminator': 'themes',
        'include_parents': False,
        'include_synonyms': False
    }),
    ('themes_tree', ThesaurusTermsListFieldIndex, {
        'interface': IThemesInfo,
        'discriminator': 'themes',
        'include_parents': True,
        'include_synonyms': False
    }),
    ('themes_all', ThesaurusTermsListFieldIndex, {
        'interface': IThemesInfo,
        'discriminator': 'themes',
        'include_parents': True,
        'include_synonyms': True
    }),
    ('collections', ThesaurusTermsListFieldIndex, {
        'interface': ICollectionsInfo,
        'discriminator': 'collections',
        'include_parents': False,
        'include_synonyms': False
    })
]


#
# Checker for required shared tables
#

def check_required_tables(site, tables=REQUIRED_TABLES, registry=None):
    """Check for required reference tables

    :param site: site root to check for tables
    :param tables: iterable of reference tables to check or create; each occurrence is a tuple of three elements
        containing configuration attribute name, default table name and table factory
    :param registry: optional registry object

    For each required table, the checking process is:
    - get table name from current site configuration, or from configuration settings using
        'pyams_content.config.{table_name}' parameter where 'table_name' is the attribute name given as first
        tuple argument
    - if table doesn't exists, create table using given factory and given table name, and store this name into
        current site configuration
    """

    def get_tables_manager():
        # check references tables manager
        tables_name = tools_configuration.tables_name or \
                      registry.settings.get('pyams_content.config.tables_name', 'references')
        if tables_name not in site:
            logger.info("Creating references table manager...")
            manager = ReferencesManager()
            registry.notify(ObjectCreatedEvent(manager))
            manager.title = {
                'en': "References tables",
                'fr': "Tables de références"
            }
            manager.short_name = {
                'en': "References tables",
                'fr': "Tables de références"
            }
            site[tables_name] = manager
            tools_configuration.tables_name = tables_name
        else:
            manager = site[tables_name]
        return manager

    tools_configuration = ISiteRootToolsConfiguration(site, None)
    if tools_configuration is not None:
        if registry is None:
            registry = get_global_registry()
        tables_manager = get_tables_manager()
        for attr, name, factory in tables:
            table_name = getattr(tools_configuration, attr, None) or \
                         registry.settings.get('pyams_content.config.{0}'.format(attr), name)
            if table_name not in tables_manager:
                table = factory()
                logger.info("Creating table {0!r}...".format(table))
                registry.notify(ObjectCreatedEvent(table))
                tables_manager[table_name] = table
                setattr(tools_configuration, attr, table_name)


#
# Checker for required shared tools
#

def check_required_tools(site, config_interface, tools):
    """Check for required shared tools

    :param site: site root to check for tools
    :param config_interface: site configuration interface; configuration attributes are updated
        when tools are created
    :param tools: iterable of shared tools to check or create; each occurrence is a tuple of two elements
        containing the tool name and it's factory interface.

    For each required tool, the checking process is:
    - get factory name from settings; the settings parameter name is 'pyams_content.config.{name}_tool_factory',
        where 'name' is the first element of tool's tuple configuration
    - if factory name is set as 'None' or '--', tool checking and creation is skipped
    - otherwise, we look for requested tool
    - if there is no shared tool with given name, we look for factory: this can be given as dotted Python name in
        configuration settings, as a named adapter from site manager to given factory interface (with name set as
        requested tool name), as an anonymous adapter from site manager to given factory interface, or as a utility
        providing given factory interface
    - if factory is found, the shared tool is created and it's name is stored into current site configuration
    """

    def get_tools_manager(site, config, registry=None):
        """Check for shared tools manager"""
        if registry is None:
            registry = get_global_registry()
        name = config.tools_name or registry.settings.get('pyams_content.config.tools_name', 'tools')
        if name not in site:
            manager = SharedToolContainer()
            registry.notify(ObjectCreatedEvent(manager))
            manager.title = {
                'en': "Shared tools",
                'fr': "Outils partagés"
            }
            manager.short_name = {
                'en': "Shared tools",
                'fr': "Outils partagés"
            }
            manager.navigation_name = {
                'en': "Shared tools",
                'fr': "Outils partagés"
            }
            site[name] = manager
            config.tools_name = name
        else:
            manager = site[name]
        return manager

    def get_required_tools(manager, config, tools=REQUIRED_TOOLS, registry=None):
        """Create required shared tools"""
        if registry is None:
            registry = get_global_registry()
        for name, factory_interface in tools:
            factory = registry.settings.get('pyams_content.config.{name}_tool_factory'.format(name=name))
            if (factory is None) or (factory.upper() not in ('NONE', '--')):
                attr_name = '{name}_tool_name'.format(name=name)
                tool_name = getattr(config, attr_name, None) or \
                            registry.settings.get('pyams_content.config.{name}'.format(name=attr_name), name)
                if tool_name not in manager:
                    if factory is not None:
                        factory = DottedNameResolver().resolve(factory)
                    if factory is None:
                        factory = registry.queryAdapter(manager, factory_interface, name=name)
                        if factory is None:
                            factory = registry.queryAdapter(manager, factory_interface)
                            if factory is None:
                                factory = registry.queryUtility(factory_interface)
                    if factory is not None:
                        tool = factory()
                        registry.notify(ObjectCreatedEvent(tool))
                        manager[tool_name] = tool
                        setattr(config, attr_name, tool_name)

    configuration = config_interface(site, None)
    if configuration is not None:
        # check shared tools manager
        manager = get_tools_manager(site, configuration)
        # check for shared tools
        get_required_tools(manager, configuration, tools)


def get_required_indexes():
    """Get list of required indexes based on lexicon settings"""
    indexes = REQUIRED_INDEXES
    registry = get_global_registry()
    for code, language in map(lambda x: x.split(':'),
                              registry.settings.get('pyams_content.lexicon.languages', 'en:english').split()):
        indexes.append((
            'title:{0}'.format(code),
            I18nTextIndexWithInterface, {
                'language': code,
                'interface': IBaseContent,
                'discriminator': 'title',
                'lexicon': lambda: get_fulltext_lexicon(language)
            }
        ))
    return indexes


@utility_config(name='PyAMS content', provides=ISiteGenerations)
class WebsiteGenerationsChecker(object):
    """PyAMS content package generations checker"""

    order = 100
    generation = 7

    def evolve(self, site, current=None):
        """Check for required utilities, tables and tools"""
        check_required_utilities(site, REQUIRED_UTILITIES)
        check_required_indexes(site, get_required_indexes())
        check_required_tables(site, REQUIRED_TABLES)
        check_required_tools(site, ISiteRootToolsConfiguration, REQUIRED_TOOLS)

        if not current:
            current = 1
        for generation in range(current, self.generation):
            module_name = 'pyams_content.generations.evolve{}'.format(generation)
            module = import_module(module_name)
            module.evolve(site)
