#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json

from pyramid.decorator import reify
from pyramid.exceptions import NotFound
from pyramid.view import view_config
from z3c.form import field
from z3c.table.interfaces import IColumn, IValues
from zope.interface import implementer

from pyams_content.features.share.interfaces import ISocialShareInfo, ISocialShareManager, \
    ISocialShareManagerTarget
from pyams_content.features.share.zmi.interfaces import ISocialShareMenu
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION
from pyams_content.zmi import pyams_content
from pyams_form.form import ajax_config
from pyams_form.group import NamedWidgetsGroup
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.help import ContentHelp
from pyams_skin.interfaces import IContentHelp, IPageHeader
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.page import DefaultPageHeaderAdapter
from pyams_skin.table import AttributeSwitcherColumn, BaseTable, I18nColumn, I18nValueColumn, \
    SorterColumn, TrashColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogEditForm
from pyams_zmi.interfaces.menu import ISiteManagementMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import ContainerAdminView

__docformat__ = 'restructuredtext'

from pyams_content import _


@viewlet_config(name='social-share.menu', context=ISocialShareManagerTarget, layer=IPyAMSLayer,
                manager=ISiteManagementMenu, permission=MANAGE_SITE_ROOT_PERMISSION, weight=25)
@implementer(ISocialShareMenu)
class SocialShareMenu(MenuItem):
    """Social network share menu"""

    label = _("Network shares...")
    icon_class = 'fa-share-alt'
    url = '#social-share.html'


class SocialShareContainerTable(BaseTable):
    """Social network share items container table"""

    prefix = 'social'

    hide_header = True
    sortOn = None

    cssClasses = {'table': 'table table-bordered table-striped table-hover table-tight table-dnd'}

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-plugins': 'pyams_content',
            'data-ams-plugin-pyams_content-src': get_resource_path(pyams_content),
            'data-ams-location': absolute_url(ISocialShareManager(self.context), self.request),
            'data-ams-tablednd-drag-handle': 'td.sorter',
            'data-ams-tablednd-drop-target': 'set-social-items-order.json',
            'data-ams-active-icon-on': 'fa fa-fw fa-check-square-o',
            'data-ams-active-icon-off': 'fa fa-fw fa-square-o txt-color-silver opacity-75'
        })
        attributes.setdefault('td', {}).update({
            'data-ams-attribute-switcher': 'switch-item-activity.json',
            'data-ams-switcher-attribute-name': 'active'
        })
        return attributes

    @reify
    def values(self):
        return list(super().values)

    def render(self):
        if not self.values:
            translate = self.request.localizer.translate
            return translate(_("No currently defined social network share item."))
        return super().render()


@adapter_config(context=(ISocialShareManagerTarget, IPyAMSLayer, SocialShareContainerTable),
                provides=IValues)
class SocialShareContainerValues(ContextRequestViewAdapter):
    """Social network share items container values"""

    @property
    def values(self):
        return ISocialShareManager(self.context).values()


@adapter_config(name='sorter',
                context=(ISocialShareManagerTarget, IPyAMSLayer, SocialShareContainerTable),
                provides=IColumn)
class SocialShareContainerSorterColumn(SorterColumn):
    """Social network share items container sorter column"""


@view_config(name='set-social-items-order.json', context=ISocialShareManager,
             request_type=IPyAMSLayer, permission=MANAGE_SITE_ROOT_PERMISSION,
             renderer='json', xhr=True)
def set_social_items_order(request):
    """Update social network share items order"""
    order = list(map(str, json.loads(request.params.get('names'))))
    request.context.updateOrder(order)
    return {'status': 'success'}


@adapter_config(name='enable-disable',
                context=(ISocialShareManagerTarget, IPyAMSLayer, SocialShareContainerTable),
                provides=IColumn)
class SocialShareContainerEnablerColumn(AttributeSwitcherColumn):
    """Social network share items container enabler switcher column"""

    switch_attribute = 'active'

    on_icon_class = 'fa fa-fw fa-check-square-o'
    off_icon_class = 'fa fa-fw fa-square-o txt-color-silver opacity-75'

    icon_hint = _("Enable/disable item")

    weight = 6


@view_config(name='switch-item-activity.json', context=ISocialShareManager,
             request_type=IPyAMSLayer, permission=MANAGE_SITE_ROOT_PERMISSION,
             renderer='json', xhr=True)
def switch_social_item_activity(request):
    """Switch social item activity"""
    container = ISocialShareManager(request.context)
    item = container.get(str(request.params.get('object_name')))
    if item is None:
        raise NotFound()
    item.active = not item.active
    return {
        'status': 'success',
        'on': item.active
    }


@adapter_config(name='name',
                context=(ISocialShareManagerTarget, IPyAMSLayer, SocialShareContainerTable),
                provides=IColumn)
class SocialShareContainerNameColumn(I18nColumn, I18nValueColumn):
    """Social network share items container name column"""

    _header = _("Label")
    attrName = 'label'
    weight = 10


@adapter_config(name='trash',
                context=(ISocialShareManagerTarget, IPyAMSLayer, SocialShareContainerTable),
                provides=IColumn)
class SocialShareContainerTrashColumn(TrashColumn):
    """Social network share items container trash column"""

    permission = MANAGE_SITE_ROOT_PERMISSION


@pagelet_config(name='social-share.html', context=ISocialShareManagerTarget, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
class SocialShareContainerView(ContainerAdminView):
    """Social network share items container view"""

    title = _("Social networks share")
    table_class = SocialShareContainerTable


@adapter_config(context=(ISocialShareManagerTarget, IAdminLayer, SocialShareContainerView),
                provides=IPageHeader)
class SocialShareContainerViewHeaderAdapter(DefaultPageHeaderAdapter):
    """Social network share items container view header adapter"""

    icon_class = 'fa fa-fw fa-share-alt'


@adapter_config(context=(ISocialShareManagerTarget, IAdminLayer, SocialShareContainerView),
                provides=IContentHelp)
class SocialShareContainerHelpAdapter(ContentHelp):
    """Social network share items container help adapter"""

    header = _("Social networks share")
    message = _("""Social networks share items are used to define share options available on your contents.
    
**WARNING**: don't forget to include a toolbox in your presentation template to display social networks shares!!
""")
    message_format = 'rest'


#
# General social share info
#

@viewlet_config(name='social-share-info.menu', context=ISocialShareManagerTarget,
                layer=IPyAMSLayer, manager=ISocialShareMenu,
                permission=MANAGE_SITE_ROOT_PERMISSION, weight=10)
class SocialShareInfoMenu(MenuItem):
    """Social share info menu"""

    label = _("Social networks info...")
    icon_class = 'fa-twitter'
    url = 'social-share-info.html'
    modal_target = True


@pagelet_config(name='social-share-info.html', context=ISocialShareManagerTarget,
                layer=IPyAMSLayer, permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='social-share-info.json', context=ISocialShareManagerTarget, layer=IPyAMSLayer)
class SocialShareInfoEditForm(AdminDialogEditForm):
    """Social share info properties edit form"""

    legend = _("Edit social networks properties")

    fields = field.Fields(ISocialShareInfo)

    label_css_class = 'control-label col-md-4'
    input_css_class = 'col-md-8'

    def updateGroups(self):
        self.add_group(NamedWidgetsGroup(self, 'twitter', self.widgets,
                                         ('twitter_account', 'twitter_creator_account'),
                                         legend=_("Twitter account"),
                                         css_class='inner'))
        self.add_group(NamedWidgetsGroup(self, 'facebook', self.widgets,
                                         ('fb_account', 'fb_app_id'),
                                         legend=_("Facebook account"),
                                         css_class='inner'))
        super(SocialShareInfoEditForm, self).updateGroups()
