#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from zope.container.contained import Contained
from zope.container.ordered import OrderedContainer
from zope.location import locate
from zope.location.interfaces import ISublocations
from zope.schema.fieldproperty import FieldProperty
from zope.traversing.interfaces import ITraversable

from pyams_catalog.utils import index_object
from pyams_content.features.share import ISocialShareItem
from pyams_content.features.share.interfaces import ISocialShareInfo, ISocialShareManager, ISocialShareManagerTarget, \
    SOCIAL_SHARE_INFO_KEY, SOCIAL_SHARE_MANAGER_KEY
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config


@factory_config(ISocialShareInfo)
class SocialShareInfo(Persistent, Contained):
    """Social network share general info"""

    twitter_account = FieldProperty(ISocialShareInfo['twitter_account'])
    twitter_creator_account = FieldProperty(ISocialShareInfo['twitter_creator_account'])
    fb_account = FieldProperty(ISocialShareInfo['fb_account'])
    fb_app_id = FieldProperty(ISocialShareInfo['fb_app_id'])


@adapter_config(context=ISocialShareManagerTarget, provides=ISocialShareInfo)
def social_share_info_factory(context):
    """Social network general info factory"""
    return get_annotation_adapter(context, SOCIAL_SHARE_INFO_KEY, ISocialShareInfo)


@factory_config(ISocialShareManager)
class SocialShareManager(OrderedContainer):
    """Social network share links manager"""

    last_id = 1

    def append(self, value, notify=True):
        key = str(self.last_id)
        if not notify:
            # pre-locate item to avoid multiple notifications
            locate(value, self, key)
        self[key] = value
        self.last_id += 1
        if not notify:
            # make sure that item is correctly indexed
            index_object(value)

    def get_active_items(self):
        yield from filter(lambda x: ISocialShareItem(x).active, self.values())


@adapter_config(context=ISocialShareManagerTarget, provides=ISocialShareManager)
def social_share_manager_factory(context):
    """Social network share manager factory"""
    return get_annotation_adapter(context, SOCIAL_SHARE_MANAGER_KEY, ISocialShareManager,
                                  name='++social-share++')


@adapter_config(name='social-share', context=ISocialShareManagerTarget, provides=ITraversable)
class SocialShareManagerNamespace(ContextAdapter):
    """Social network share manager ++social-share++ namespace"""

    def traverse(self, name, furtherpath=None):
        return ISocialShareManager(self.context)


@adapter_config(name='social-share', context=ISocialShareManagerTarget, provides=ISublocations)
class SocialShareManagerSublocations(ContextAdapter):
    """Social network share manager sub-locations adapter"""

    def sublocations(self):
        return ISocialShareManager(self.context).values()
