#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from pyramid.encode import url_quote
from zope.container.contained import Contained
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from pyams_content.features.share.interfaces import ISocialShareItem
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION
from pyams_content.reference.pictograms import IPictogramTable
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n
from pyams_utils.adapter import ContextAdapter, adapter_config
from pyams_utils.registry import query_utility
from pyams_utils.url import canonical_url
from pyams_utils.zodb import volatile_property


@implementer(ISocialShareItem)
class SocialShareItem(Persistent, Contained):
    """Social network share item"""

    active = FieldProperty(ISocialShareItem['active'])
    label = FieldProperty(ISocialShareItem['label'])
    target_url = FieldProperty(ISocialShareItem['target_url'])
    _pictogram_name = FieldProperty(ISocialShareItem['pictogram_name'])

    @property
    def pictogram_name(self):
        return self._pictogram_name

    @pictogram_name.setter
    def pictogram_name(self, value):
        if value != self._pictogram_name:
            self._pictogram_name = value
            del self.pictogram

    @volatile_property
    def pictogram(self):
        table = query_utility(IPictogramTable)
        if table is not None:
            return table.get(self._pictogram_name)

    def get_url(self, context, request):
        url = II18n(self).query_attribute('target_url', request=request)
        return url.format(url=url_quote(canonical_url(context, request)),
                          title=url_quote(II18n(context).query_attribute('title', request=request)))


@adapter_config(context=ISocialShareItem, provides=IFormContextPermissionChecker)
class SocialShareItemPermissionChecker(ContextAdapter):
    """Social share item permission checker"""

    edit_permission = MANAGE_SITE_ROOT_PERMISSION
