#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.association.container import AssociationContainer
from pyams_content.component.association.interfaces import IAssociationInfo
from pyams_content.component.links import IInternalLink, InternalLink, InternalLinkAssociationInfoAdapter, \
    InternalReferenceMixin
from pyams_content.features.menu.interfaces import IDynamicMenu, IMenu, IMenuLink, IMenusContainer, \
    IMenusContainerTarget
from pyams_content.reference.pictograms import IPictogramTable
from pyams_content.shared.common import IWfSharedContent
from pyams_content.shared.site.interfaces import ISiteContainer
from pyams_i18n.interfaces import II18n
from pyams_sequence.interfaces import ISequentialIdInfo, ISequentialIdTarget
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import query_utility
from pyams_utils.request import check_request
from pyams_utils.url import relative_url
from pyams_utils.zodb import volatile_property
from pyams_workflow.interfaces import IWorkflowPublicationInfo
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'


#
# Menus classes
#

@factory_config(IMenu)
class Menu(AssociationContainer, InternalReferenceMixin):
    """Associations menu"""

    visible = FieldProperty(IMenu['visible'])
    title = FieldProperty(IMenu['title'])
    reference = FieldProperty(IMenu['reference'])
    dynamic_menu = FieldProperty(IMenu['dynamic_menu'])
    force_canonical_url = FieldProperty(IMenu['force_canonical_url'])
    _pictogram_name = FieldProperty(IMenu['pictogram_name'])

    @property
    def pictogram_name(self):
        return self._pictogram_name

    @pictogram_name.setter
    def pictogram_name(self, value):
        if value != self._pictogram_name:
            self._pictogram_name = value
            del self.pictogram

    @volatile_property
    def pictogram(self):
        table = query_utility(IPictogramTable)
        if table is not None:
            return table.get(self._pictogram_name)

    def is_visible(self, request=None):
        if not self.reference:
            return True
        target = self.get_target()
        if target is not None:
            publication_info = IWorkflowPublicationInfo(target, None)
            if publication_info is not None:
                return publication_info.is_visible(request)

    def get_visible_items(self, request=None):
        if self.dynamic_menu and ISiteContainer.providedBy(self.target):
            for item in filter(None,
                               map(lambda x: IDynamicMenu(x, None),
                                   self.target.get_visible_items(request))):
                if IInternalLink.providedBy(item):
                    item.force_canonical_url = self.force_canonical_url
                yield item
        yield from super(Menu, self).get_visible_items(request)

    def get_url(self, request=None, view_name=None):
        target = self.get_target()
        if target is not None:
            if request is None:
                request = check_request()
            return relative_url(target, request, view_name=view_name)
        return ''


@adapter_config(context=IMenu, provides=IAssociationInfo)
class MenuAssociationInfoAdapter(InternalLinkAssociationInfoAdapter):
    """Menu association adapter"""

    @property
    def user_header(self):
        description = None
        target = self.context.get_target()
        if (target is not None) and hasattr(target, 'header'):
            description = II18n(target).query_attribute('header')
        return description


@factory_config(IMenusContainer)
class MenusContainer(AssociationContainer):
    """Associations menus container"""

    def get_visible_items(self, request=None):
        for menu in filter(lambda x: IMenu(x).visible, self.values()):
            if IAdminLayer.providedBy(request) or menu.is_visible(request):
                yield menu


@adapter_config(context=ISequentialIdTarget, provides=IDynamicMenu)
@adapter_config(context=IWfSharedContent, provides=IDynamicMenu)
def content_dynamic_menu_factory(context):
    """Shared content dynamic menu factory"""
    sequence = ISequentialIdInfo(context, None)
    if sequence is not None:
        result = InternalLink()
        result.reference = sequence.hex_oid
        return result
