#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""
from enum import Enum

from pyams_content.features.json.interfaces import IJSONConverter
from pyams_file.interfaces import IThumbnails
from pyams_i18n.interfaces import II18n, INegotiator
from pyams_utils.adapter import ContextRequestAdapter
from pyams_utils.interfaces.text import IHTMLRenderer
from pyams_utils.registry import get_global_registry, get_utility
from pyams_utils.request import check_request
from pyams_utils.url import absolute_url

__docformat__ = 'restructuredtext'


class JSONBaseConverter(ContextRequestAdapter):
    """JSON base converter"""

    def to_json(self, params=None):
        """JSON converter"""
        if params is None:
            params = {}
        lang = params.get('lang')
        if not lang:
            negotiator = get_utility(INegotiator)
            params['lang'] = negotiator.server_language
        result = self.convert_content(params)
        registry = get_global_registry()
        target = self.conversion_target
        if target is None:
            return result
        for name, converter in registry.getAdapters((target, self.request), IJSONConverter):
            if not name:  # exclude this default adapter
                continue
            if (('included' in params) and (name not in params['included'].split(','))) or \
                    (('excluded' in params) and (name in params['excluded'].split(','))):
                continue
            output = converter.to_json(params)
            if not output:
                continue
            if converter.is_inner:
                result.update({name: output})
            else:
                result.update(output)
        return result

    @property
    def conversion_target(self):
        return self.context

    def convert_content(self, params):
        """Base context converter"""
        return {}

    @staticmethod
    def get_attribute(context, attr, result, name=None):
        """Get standard attribute"""
        if not hasattr(context, attr):
            return
        if name is None:
            name = attr
        value = getattr(context, attr)
        if value or (value is False):
            result[name] = value

    @staticmethod
    def get_i18n_attribute(context, attr, lang, result, name=None):
        """Get localized attribute"""
        if not hasattr(context, attr):
            return
        if name is None:
            name = attr
        if lang:
            value = II18n(context).query_attribute(attr, lang=lang)
            if value:
                result[name] = value

    def get_html_attribute(self, context, attr, lang, result, name=None):
        """Get HTML attribute"""
        if not hasattr(context, attr):
            return
        if name is None:
            name = attr
        if lang:
            value = II18n(context).query_attribute(attr, lang=lang)
            if value:
                renderer = self.request.registry.queryMultiAdapter((value, self.request),
                                                                   IHTMLRenderer,
                                                                   name='oid_to_href')
                if renderer is not None:
                    result[name] = renderer.render()

    def get_list_attribute(self, items, params, result, name):
        """Get list as inner """
        registry = get_global_registry()
        values = []
        for item in items:
            converter = registry.queryMultiAdapter((item, self.request), IJSONConverter)
            if converter is not None:
                value = converter.to_json(params)
                if value:
                    values.append(value)
        if values:
            result[name] = values

    def get_file_url(self, context, attr, params):
        """Get file URL"""
        file = getattr(context, attr, None)
        if isinstance(file, dict):
            file = file.get(params.get('lang'))
        if not file:
            return None
        return absolute_url(file, self.request)

    def get_image_url(self, context, attr, params):
        """Get image URL"""
        image = getattr(context, attr, None)
        if isinstance(image, dict):
            image = image.get(params.get('lang'))
        if not image:
            return None
        thumbnails = IThumbnails(image, None)
        if thumbnails is not None:
            display_name = params.get('display_name')
            if isinstance(display_name, Enum):
                display_name = display_name.value
            display_size = params.get('display_size', 'w800')
            if display_name:
                thumbnail = thumbnails.get_thumbnail('{}:{}'.format(display_name, display_size))
            else:
                thumbnail = thumbnails.get_thumbnail(display_size)
            return absolute_url(thumbnail, self.request)
        return absolute_url(image, self.request)


def to_json(context, request=None, params=None):
    """Convert provided context to JSON"""
    if request is None:
        request = check_request()
    registry = request.registry
    converter = registry.queryMultiAdapter((context, request), IJSONConverter)
    if converter is None:
        return {}
    params = params or {}
    return converter.to_json(params)
