#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from pyams_cache.beaker import get_cache
from pyramid.events import subscriber
from zope.lifecycleevent.interfaces import IObjectModifiedEvent
from zope.location import Location
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary
from zope.traversing.interfaces import ITraversable

from pyams_content.features.header.interfaces import HEADER_RENDERERS, HEADER_RENDERER_SETTINGS_KEY, \
    HEADER_SETTINGS_KEY, IHeaderRenderer, IHeaderRendererSettings, IHeaderSettings, IHeaderTarget
from pyams_content.features.renderer import RenderedContentMixin
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_portal.interfaces import MANAGE_TEMPLATE_PERMISSION
from pyams_portal.portlet import PORTLETS_CACHE_NAME, PORTLETS_CACHE_REGION
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.inherit import BaseInheritInfo, InheritedFieldProperty
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


__docformat__ = 'restructuredtext'


@factory_config(IHeaderSettings)
class HeaderSettings(BaseInheritInfo, RenderedContentMixin, Persistent, Location):
    """Header settings persistent class"""

    target_interface = IHeaderTarget
    adapted_interface = IHeaderSettings

    _renderer = FieldProperty(IHeaderSettings['renderer'])
    renderer = InheritedFieldProperty(IHeaderSettings['renderer'])

    renderer_interface = IHeaderRenderer

    def get_renderer(self, request=None, renderer_name=None):
        if request is None:
            request = check_request()
        if not renderer_name:
            renderer_name = request.params.get('form.widgets.renderer')
            if renderer_name is None:
                renderer_name = self.renderer or 'hidden'
        parent = get_parent(self, IHeaderTarget)
        return request.registry.queryMultiAdapter((parent, request), self.renderer_interface,
                                                  name=renderer_name)

    def get_settings(self, renderer_name=None):
        renderer = self.get_renderer(renderer_name=renderer_name)
        if (renderer is None) or (renderer.settings_interface is None):
            return None
        settings_key = HEADER_RENDERER_SETTINGS_KEY.format(renderer.settings_key)
        return get_annotation_adapter(self, settings_key,
                                      factory=lambda: IHeaderRendererSettings(renderer),
                                      name='++settings++{0}'.format(renderer.name))

    @property
    def settings(self):
        return self.get_settings()


@adapter_config(context=IHeaderTarget, provides=IHeaderSettings)
def header_settings_factory(context):
    """Header settings factory"""
    return get_annotation_adapter(context, HEADER_SETTINGS_KEY, IHeaderSettings,
                                  name='++header++')


@subscriber(IObjectModifiedEvent, context_selector=IHeaderSettings)
def handle_modified_header_settings(event):
    """Clear cache if modified header settings"""
    renderer = event.object.get_renderer()
    portlets_cache = get_cache(PORTLETS_CACHE_REGION, PORTLETS_CACHE_NAME)
    portlets_cache.remove(renderer.cache_key)


@adapter_config(name='header', context=IHeaderTarget, provides=ITraversable)
class HeaderTargetNamespace(ContextAdapter):
    """Header target '++header++' namespace traverser"""

    def traverse(self, name, furtherpath=None):
        return IHeaderSettings(self.context)


@adapter_config(context=IHeaderSettings, provides=IHeaderRendererSettings)
def header_settings_renderer_settings_factory(context):
    """Header settings renderer settings factory"""
    return context.settings


@adapter_config(name='settings', context=IHeaderSettings, provides=ITraversable)
class HeaderSettingsRendererSettingsNamespace(ContextAdapter):
    """Header settings '++settings++' namespace traverser"""

    def traverse(self, name, furtherpath=None):
        if name:
            return self.context.get_settings(renderer_name=name)
        else:
            return IHeaderRendererSettings(self.context)


@adapter_config(context=IHeaderTarget, provides=IHeaderRendererSettings)
def header_target_renderer_settings_factory(context):
    """Header target renderer settings factory"""
    settings = IHeaderSettings(context)
    return IHeaderRendererSettings(settings, None)


@adapter_config(context=IHeaderRendererSettings, provides=IFormContextPermissionChecker)
class HeaderRendererSettingsPermissionChecker(ContextAdapter):
    """Header renderer settings permission checker"""

    edit_permission = MANAGE_TEMPLATE_PERMISSION


@vocabulary_config(name=HEADER_RENDERERS)
class HeaderRendererVocabulary(SimpleVocabulary):
    """Header renderers vocabulary"""

    def __init__(self, context=None):
        request = check_request()
        if context is None:
            context = request.context
        translate = request.localizer.translate
        registry = request.registry
        target = get_parent(context, IHeaderTarget)
        terms = [SimpleTerm(name, title=translate(adapter.label))
                 for name, adapter in sorted(registry.getAdapters((target, request), IHeaderRenderer),
                                             key=lambda x: x[1].weight)]
        super(HeaderRendererVocabulary, self).__init__(terms)
