#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from pyams_content.features.footer.interfaces import IFooterTarget, IFooterSettings, IFooterRenderer, \
    IFooterRendererSettings
from pyams_form.interfaces.form import IWidgetForm, IUncheckedEditFormButtons, IInnerSubForm, \
    IWidgetsSuffixViewletsManager, IFormHelp
from pyams_portal.interfaces import MANAGE_TEMPLATE_PERMISSION
from pyams_portal.zmi.interfaces import IPortalContextTemplatePropertiesMenu
from pyams_skin.interfaces import IInnerPage
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.interfaces.inherit import IInheritInfo
from pyams_zmi.layer import IAdminLayer
from z3c.form.interfaces import INPUT_MODE

# import packages
from pyams_content.zmi import pyams_content
from pyams_form.form import AJAXEditForm
from pyams_form.group import NamedWidgetsGroup
from pyams_form.help import FormHelp
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.viewlet.menu import MenuItem
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config, Viewlet
from pyams_zmi.form import AdminEditForm, InnerAdminEditForm
from pyramid.exceptions import NotFound
from pyramid.response import Response
from pyramid.view import view_config
from z3c.form import field, button
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
from zope.interface import implementer, alsoProvides, Interface

from pyams_content import _


@viewlet_config(name='footer-settings.menu', context=IFooterTarget, layer=IPyAMSLayer,
                manager=IPortalContextTemplatePropertiesMenu, permission=MANAGE_TEMPLATE_PERMISSION, weight=102)
class FooterSettingsMenu(MenuItem):
    """Footer settings menu"""

    label = _("Page footer")
    url = '#footer-settings.html'


class IFooterSettingsGroup(Interface):
    """Footer settings group marker interface"""


@pagelet_config(name='footer-settings.html', context=IFooterTarget, layer=IPyAMSLayer,
                permission=MANAGE_TEMPLATE_PERMISSION)
@implementer(IWidgetForm, IInnerPage)
class FooterSettingsEditForm(AdminEditForm):
    """Footer settings edit form"""

    @property
    def title(self):
        return self.context.title

    legend = _("Edit footer settings")

    def getContent(self):
        return IFooterSettings(self.context)

    @property
    def fields(self):
        if self.getContent().can_inherit:
            fields = field.Fields(IFooterSettings).select('no_inherit')
            fields['no_inherit'].widgetFactory = SingleCheckBoxFieldWidget
        else:
            fields = field.Fields(Interface)
        return fields

    edit_permission = MANAGE_TEMPLATE_PERMISSION

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IUncheckedEditFormButtons)
        else:
            return button.Buttons(Interface)

    ajax_handler = 'footer-settings.json'

    def updateGroups(self):
        if self.getContent().can_inherit:
            group = NamedWidgetsGroup(self, 'footer', self.widgets,
                                      ('no_inherit', ),
                                      legend=_("Don't inherit parent footer"),
                                      css_class='inner',
                                      switch=True,
                                      checkbox_switch=True,
                                      checkbox_mode='disable',
                                      checkbox_field=IFooterSettings['no_inherit'])
            if self.mode == INPUT_MODE:
                group.object_data = {
                    'ams-plugins': 'pyams_content',
                    'ams-plugin-pyams_content-src': get_resource_path(pyams_content),
                    'ams-plugin-pyams_content-async': False,
                    'ams-events-handlers': {
                        'ams.checker.opened': 'PyAMS_content.footer.submitEditForm',
                        'ams.checker.closed': 'PyAMS_content.footer.submitEditForm'
                    }
                }
                alsoProvides(group, IObjectData)
        else:
            group = NamedWidgetsGroup(self, 'footer', self.widgets, (), css_class='inner')
        alsoProvides(group, IFooterSettingsGroup)
        self.add_group(group)
        super(FooterSettingsEditForm, self).updateGroups()


@view_config(name='footer-settings.json', context=IFooterTarget, request_type=IPyAMSLayer,
             permission=MANAGE_TEMPLATE_PERMISSION, renderer='json', xhr=True)
class FooterSettingsAJAXEditForm(AJAXEditForm, FooterSettingsEditForm):
    """Footer settings edit form, JSON renderer"""

    def get_ajax_output(self, changes):
        output = super(FooterSettingsAJAXEditForm, self).get_ajax_output(changes) or {}
        if 'no_inherit' in changes.get(IInheritInfo, ()):
            output['status'] = 'reload'
        return output


@adapter_config(context=(Interface, IAdminLayer, FooterSettingsEditForm), provides=IFormHelp)
class FooterSettingsEditFormHelpAdapter(FormHelp):
    """Footer settings edit form help adapter"""

    def __new__(cls, context, request, view):
        if (not view.getContent().can_inherit) or (view.mode != INPUT_MODE):
            return None
        return FormHelp.__new__(cls)

    message = _("""WARNING: Footer properties are saved automatically when changing inherit mode!!""")
    message_format = 'rest'


@adapter_config(name='renderer', context=(IFooterTarget, IPyAMSLayer, IFooterSettingsGroup), provides=IInnerSubForm)
class FooterSettingsRendererEditSubform(InnerAdminEditForm):
    """Footer settings renderer edit form"""

    legend = None

    fields = field.Fields(IFooterSettings).select('renderer')
    weight = 1

    edit_permission = MANAGE_TEMPLATE_PERMISSION

    _changes = None

    def __init__(self, context, request, group):
        context = IFooterSettings(context)
        super(FooterSettingsRendererEditSubform, self).__init__(context, request, group)

    def getContent(self):
        return IFooterSettings(self.context)

    def updateWidgets(self, prefix=None):
        super(FooterSettingsRendererEditSubform, self).updateWidgets(prefix)
        if 'renderer' in self.widgets:
            widget = self.widgets['renderer']
            widget.object_data = {
                'ams-change-handler': 'MyAMS.helpers.select2ChangeHelper',
                'ams-change-stop-propagation': 'true',
                'ams-select2-helper-type': 'html',
                'ams-select2-helper-url': absolute_url(self.context, self.request,
                                                       'get-footer-settings-renderer-form.html'),
                'ams-select2-helper-argument': 'form.widgets.renderer',
                'ams-select2-helper-target': '#renderer-settings-helper'
            }
            alsoProvides(widget, IObjectData)

    def get_forms(self, include_self=True):
        if include_self and (self._changes is None) and (self.request.method == 'POST'):
            data, errors = self.extractData()
            if not errors:
                self._changes = self.applyChanges(data)
        for form in super(FooterSettingsRendererEditSubform, self).get_forms(include_self):
            yield form

    def get_ajax_output(self, changes):
        if not changes:
            changes = self._changes
        if changes:
            return {
                'status': 'success',
                'message': self.request.localizer.translate(self.successMessage)
            }
        else:
            return super(FooterSettingsRendererEditSubform, self).get_ajax_output(changes)


@adapter_config(name='footer-renderer-settings-form',
                context=(IFooterRendererSettings, IPyAMSLayer, FooterSettingsRendererEditSubform),
                provides=IInnerSubForm)
@adapter_config(name='footer-renderer-settings-form',
                context=(IFooterTarget, IPyAMSLayer, FooterSettingsAJAXEditForm),
                provides=IInnerSubForm)
class FooterSettingsRendererSettingsEditForm(InnerAdminEditForm):
    """Footer settings renderer settings edit form"""

    legend = _("Footer renderer settings")

    def __new__(cls, context, request, view=None):
        settings = IFooterRendererSettings(context, None)
        if settings is None:
            return None
        return InnerAdminEditForm.__new__(cls)

    def __init__(self, context, request, view=None):
        context = IFooterRendererSettings(context)
        super(FooterSettingsRendererSettingsEditForm, self).__init__(context, request, view)


@viewlet_config(name='footer-renderer-settings', context=IFooterSettings, layer=IPyAMSLayer,
                view=FooterSettingsRendererEditSubform, manager=IWidgetsSuffixViewletsManager)
@template_config(template='templates/renderer-settings.pt', layer=IPyAMSLayer)
class FooterSettingsRendererSettingsWidgetsSuffix(Viewlet):
    """Footer settings renderer settings viewlet"""

    def render_edit_form(self):
        settings = IFooterSettings(self.context)
        renderer = settings.get_renderer(self.request)
        if (renderer is None) or (renderer.settings_interface is None):
            return ''
        renderer_settings = IFooterRendererSettings(settings)
        form = FooterSettingsRendererSettingsEditForm(renderer_settings, self.request)
        form.update()
        return form.render()


@view_config(name='get-footer-settings-renderer-form.html', context=IFooterSettings,
             request_type=IPyAMSLayer, permission=MANAGE_TEMPLATE_PERMISSION, xhr=True)
def get_footer_settings_renderer_form(request):
    """Footer settings renderer settings form"""
    renderer_name = request.params.get('form.widgets.renderer')
    if renderer_name is None:
        raise NotFound("No provided renderer argument")
    if not renderer_name:
        renderer_name = ''
    renderer = request.registry.queryMultiAdapter((request.context, request), IFooterRenderer, name=renderer_name)
    if (renderer is None) or (renderer.settings_interface is None):
        return Response('')
    settings = IFooterSettings(request.context)
    renderer_settings = IFooterRendererSettings(settings)
    form = FooterSettingsRendererSettingsEditForm(renderer_settings, request)
    form.update()
    return Response(form.render())
