#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from pyams_cache.beaker import get_cache
from pyramid.events import subscriber
from zope.lifecycleevent.interfaces import IObjectModifiedEvent
from zope.location import Location
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary
from zope.traversing.interfaces import ITraversable

from pyams_content.features.footer.interfaces import FOOTER_RENDERERS, FOOTER_RENDERER_SETTINGS_KEY, \
    FOOTER_SETTINGS_KEY, IFooterRenderer, IFooterRendererSettings, IFooterSettings, IFooterTarget
from pyams_content.features.renderer import RenderedContentMixin
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_portal.interfaces import MANAGE_TEMPLATE_PERMISSION
from pyams_portal.portlet import PORTLETS_CACHE_NAME, PORTLETS_CACHE_REGION
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.inherit import BaseInheritInfo, InheritedFieldProperty
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


__docformat__ = 'restructuredtext'


@factory_config(IFooterSettings)
class FooterSettings(BaseInheritInfo, RenderedContentMixin, Persistent, Location):
    """Footer settings persistent class"""

    target_interface = IFooterTarget
    adapted_interface = IFooterSettings

    _renderer = FieldProperty(IFooterSettings['renderer'])
    renderer = InheritedFieldProperty(IFooterSettings['renderer'])

    renderer_interface = IFooterRenderer

    def get_renderer(self, request=None, renderer_name=None):
        if request is None:
            request = check_request()
        if not renderer_name:
            renderer_name = request.params.get('form.widgets.renderer')
            if renderer_name is None:
                renderer_name = self.renderer or 'hidden'
        parent = get_parent(self, IFooterTarget)
        return request.registry.queryMultiAdapter((parent, request), self.renderer_interface,
                                                  name=renderer_name)

    def get_settings(self, renderer_name=None):
        renderer = self.get_renderer(renderer_name=renderer_name)
        if (renderer is None) or (renderer.settings_interface is None):
            return None
        settings_key = FOOTER_RENDERER_SETTINGS_KEY.format(renderer.settings_key)
        return get_annotation_adapter(self, settings_key,
                                      factory=lambda: IFooterRendererSettings(renderer),
                                      name='++settings++{0}'.format(renderer.name))

    @property
    def settings(self):
        return self.get_settings()


@adapter_config(context=IFooterTarget, provides=IFooterSettings)
def footer_settings_factory(context):
    """Footer settings factory"""
    return get_annotation_adapter(context, FOOTER_SETTINGS_KEY, IFooterSettings,
                                  name='++footer++')


@subscriber(IObjectModifiedEvent, context_selector=IFooterSettings)
def handle_modified_footer_settings(event):
    """Clear cache if modified footer settings"""
    renderer = event.object.get_renderer()
    portlets_cache = get_cache(PORTLETS_CACHE_REGION, PORTLETS_CACHE_NAME)
    portlets_cache.remove(renderer.cache_key)


@adapter_config(name='footer', context=IFooterTarget, provides=ITraversable)
class FooterTargetNamespace(ContextAdapter):
    """Footer target '++footer++' namespace traverser"""

    def traverse(self, name, furtherpath=None):
        return IFooterSettings(self.context)


@adapter_config(context=IFooterSettings, provides=IFooterRendererSettings)
def footer_settings_renderer_settings_factory(context):
    """Footer settings renderer settings factory"""
    return context.settings


@adapter_config(name='settings', context=IFooterSettings, provides=ITraversable)
class FooterSettingsRendererSettingsNamespace(ContextAdapter):
    """Footer settings '++settings++' namespace traverser"""

    def traverse(self, name, furtherpath=None):
        if name:
            return self.context.get_settings(renderer_name=name)
        else:
            return IFooterRendererSettings(self.context)


@adapter_config(context=IFooterTarget, provides=IFooterRendererSettings)
def footer_target_renderer_settings_factory(context):
    """Footer target renderer settings factory"""
    settings = IFooterSettings(context)
    return IFooterRendererSettings(settings, None)


@adapter_config(context=IFooterRendererSettings, provides=IFormContextPermissionChecker)
class FooterRendererSettingsPermissionChecker(ContextAdapter):
    """Footer renderer settings permission checker"""

    edit_permission = MANAGE_TEMPLATE_PERMISSION


@vocabulary_config(name=FOOTER_RENDERERS)
class FooterRendererVocabulary(SimpleVocabulary):
    """Footer renderers vocabulary"""

    def __init__(self, context=None):
        request = check_request()
        if context is None:
            context = request.context
        translate = request.localizer.translate
        registry = request.registry
        target = get_parent(context, IFooterTarget)
        terms = [SimpleTerm(name, title=translate(adapter.label))
                 for name, adapter in sorted(registry.getAdapters((target, request), IFooterRenderer),
                                             key=lambda x: x[1].weight)]
        super(FooterRendererVocabulary, self).__init__(terms)
