#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from itertools import chain, tee

from persistent import Persistent
from zope.container.contained import Contained
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.links import InternalReferenceMixin
from pyams_content.features.alert.interfaces import ALERT_CONTAINER_KEY, IAlertManagerInfo
from pyams_content.root.interfaces import ISiteRoot
from pyams_content.shared.alert import ALERT_CONTENT_TYPE
from pyams_content.shared.alert.interfaces import IAlertsManager
from pyams_sequence.interfaces import ISequentialIdInfo
from pyams_sequence.reference import get_reference_target
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.list import unique_iter
from pyams_utils.registry import query_utility


CONTEXT_ALERTS = 'pyams_content.context_alerts'


@factory_config(IAlertManagerInfo)
class AlertManagerInfo(InternalReferenceMixin, Persistent, Contained):
    """Alert manager info"""

    _reference = FieldProperty(IAlertManagerInfo['reference'])
    context_view = FieldProperty(IAlertManagerInfo['context_view'])

    @property
    def reference(self):
        return self._reference

    @reference.setter
    def reference(self, value):
        self._reference = value
        del self.target

    def get_context_alerts(self, request, context=None):
        """Iterator over visible alerts associated with provided context"""
        if context is None:
            context = request.context
        view = get_reference_target(self.context_view)
        if view is not None:
            shared_alerts = view.get_results(context=context,
                                             request=request,
                                             content_type=ALERT_CONTENT_TYPE)
        else:
            shared_alerts = ()
        manager = query_utility(IAlertsManager)
        if manager is not None:
            context_alerts = manager.find_context_alerts(context, request)
        else:
            context_alerts = ()
        alerts, copied_alerts = tee(unique_iter(chain(shared_alerts, context_alerts)))
        # exclude items from future searches
        excluded_alerts = set((
            ISequentialIdInfo(alert).hex_oid
            for alert in copied_alerts
        ))
        request.annotations[CONTEXT_ALERTS] = excluded_alerts
        yield from alerts

    def get_visible_alerts(self, request):
        """Iterator over visible alerts"""

        def check_alert(alert):
            """Hide alerts matching current request context"""
            excluded_alerts = request.annotations.get(CONTEXT_ALERTS) or set()
            for target in alert.get_targets():
                if target is request.context:
                    return False
                if ISequentialIdInfo(alert).hex_oid in excluded_alerts:
                    return False
            return True

        view = self.target
        if view is None:
            return
        yield from filter(check_alert,
                          view.get_results(context=request.context,
                                           request=request,
                                           content_type=ALERT_CONTENT_TYPE))


@adapter_config(context=ISiteRoot,
                provides=IAlertManagerInfo)
def site_root_alerts_manager(context):
    return get_annotation_adapter(context, ALERT_CONTAINER_KEY, IAlertManagerInfo)
