#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.events import subscriber
from pyramid.exceptions import NotFound
from pyramid.response import Response
from pyramid.view import view_config
from z3c.form import button, field
from z3c.form.interfaces import DISPLAY_MODE, IDataExtractedEvent, INPUT_MODE
from zope.interface import Interface, Invalid, alsoProvides, implementer
from zope.schema import getFieldNamesInOrder

from pyams_content.component.paragraph.interfaces import IParagraphContainerTarget
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, BaseParagraphAJAXEditForm, \
    BaseParagraphAddForm, BaseParagraphAddMenu, BaseParagraphPropertiesEditForm, IParagraphInnerEditFormButtons, \
    get_json_paragraph_refresh_event
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerView, IParagraphInnerEditor
from pyams_content.component.video.interfaces import EXTERNAL_VIDEO_PARAGRAPH_TYPE, IExternalVideoParagraph, \
    IExternalVideoProvider, IExternalVideoSettings
from pyams_content.features.renderer.zmi.widget import RendererFieldWidget
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_form.form import ajax_config
from pyams_form.group import NamedWidgetsGroup
from pyams_form.interfaces.form import IInnerForm, IWidgetsSuffixViewletsManager
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.registry import get_utility
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import Viewlet, viewlet_config
from pyams_zmi.form import InnerAdminAddForm, InnerAdminEditForm
from pyams_zmi.interfaces import IPropertiesEditForm

from pyams_content import _


@viewlet_config(name='add-external-video.menu', context=IParagraphContainerTarget, view=IParagraphContainerView,
                layer=IPyAMSLayer, manager=IToolbarAddingMenu, weight=75)
class ExternalVideoParagraphAddMenu(BaseParagraphAddMenu):
    """External video paragraph add menu"""

    label = _("External video...")
    label_css_class = 'fa fa-fw fa-youtube-play'
    url = 'add-external-video.html'
    paragraph_type = EXTERNAL_VIDEO_PARAGRAPH_TYPE


@pagelet_config(name='add-external-video.html', context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-external-video.json', context=IParagraphContainerTarget, layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class ExternalVideoParagraphAddForm(BaseParagraphAddForm):
    """External video paragraph add form"""

    legend = _("Add new external video...")
    dialog_class = 'modal-large'
    icon_css_class = 'fa fa-fw fa-youtube-play'

    content_interface = IExternalVideoParagraph
    fields = field.Fields(content_interface).select('title', 'author', 'description', 'renderer', 'provider_name')

    def updateWidgets(self, prefix=None):
        super(ExternalVideoParagraphAddForm, self).updateWidgets(prefix)
        if 'provider_name' in self.widgets:
            widget = self.widgets['provider_name']
            widget.required = True
            widget.prompt = True
            widget.promptMessage = _("Select video provider...")
            widget.object_data = {
                'ams-change-handler': 'MyAMS.helpers.select2ChangeHelper',
                'ams-change-stop-propagation': 'true',
                'ams-select2-helper-type': 'html',
                'ams-select2-helper-url': absolute_url(self.context, self.request,
                                                       'get-video-provider-settings-add-form.html'),
                'ams-select2-helper-argument': 'provider_name',
                'ams-select2-helper-target': '#video-settings-helper'
            }
            alsoProvides(widget, IObjectData)

    def update_content(self, content, data):
        changes = super(ExternalVideoParagraphAddForm, self).update_content(content, data)
        settings = IExternalVideoSettings(content, None)
        if settings is not None:
            provider = content.get_provider()
            form = InnerAdminEditForm(settings, self.request)
            form.prefix = self.prefix
            form.edit_permission = MANAGE_CONTENT_PERMISSION
            form.fields = field.Fields(provider.settings_interface)
            form.update()
            settings_data, errors = form.extractData()
            if not errors:
                changes.update(form.update_content(settings, settings_data))
        return changes


@subscriber(IDataExtractedEvent, form_selector=ExternalVideoParagraphAddForm)
def handle_video_paragraph_add_form_data_extraction(event):
    """Handle provider name data extraction"""
    data = event.data
    if not data.get('provider_name'):
        event.form.widgets.errors += (Invalid(_("Video provider is required")),)


@viewlet_config(name='external-video-settings', context=Interface, layer=IPyAMSLayer,
                view=ExternalVideoParagraphAddForm, manager=IWidgetsSuffixViewletsManager)
@template_config(template='templates/video-settings.pt', layer=IPyAMSLayer)
class VideoSettingsWidgetsSuffix(Viewlet):
    """External video settings edit form widgets suffix"""


class ExternalVideoProviderSettingsAddForm(InnerAdminAddForm):
    """External video provider settings add form"""

    def __init__(self, context, request, provider):
        super(ExternalVideoProviderSettingsAddForm, self).__init__(context, request)
        self.provider = provider

    def updateGroups(self):
        if 'integration_code' in self.widgets:  # custom video provider
            self.add_group(NamedWidgetsGroup(self, 'integration_code', self.widgets, ('integration_code',),
                                             bordered=False))
        elif 'video_id' in self.widgets:
            self.add_group(NamedWidgetsGroup(self, 'video_id', self.widgets, ('video_id',), bordered=False))
        field_names = getFieldNamesInOrder(self.provider.settings_interface)
        if len(field_names) > 1:
            self.add_group(NamedWidgetsGroup(self, 'provider_group', self.widgets,
                                             field_names[1:],
                                             bordered=False,
                                             legend=_("Other settings"),
                                             css_class="inner switcher padding-right-10 no-y-padding",
                                             switch=True,
                                             display_mode='auto'))
        super(ExternalVideoProviderSettingsAddForm, self).updateGroups()


@view_config(name='get-video-provider-settings-add-form.html', context=IParagraphContainerTarget,
             request_type=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION, xhr=True)
def external_video_provider_settings_add_form(request):
    """External video provider settings form"""
    provider_name = request.params.get('provider_name')
    if provider_name is None:
        raise NotFound("No provided provider_name argument")
    elif (not provider_name) or (provider_name == '--NOVALUE--'):
        return Response('')
    else:
        provider = get_utility(IExternalVideoProvider, name=provider_name)
        form = ExternalVideoProviderSettingsAddForm(request.context, request, provider)
        form.prefix = ExternalVideoParagraphAddForm.prefix
        form.legend = request.localizer.translate(_("Video provider settings"))
        form.label_css_class = 'control-label col-md-4'
        form.input_css_class = 'col-md-8'
        form.fields = field.Fields(provider.settings_interface)
        form.update()
        return Response(form.render())


@pagelet_config(name='properties.html', context=IExternalVideoParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IExternalVideoParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
class ExternalVideoParagraphPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """External video paragraph properties edit form"""

    prefix = 'externalvideo_properties.'

    legend = _("Edit video properties")

    content_interface = IExternalVideoParagraph

    @property
    def fields(self):
        fields = super(ExternalVideoParagraphPropertiesEditForm, self).fields \
            .select('title', 'author', 'description', 'renderer', 'provider_name')
        provider = self.context.get_provider()
        if provider is not None:
            fields += field.Fields(provider.settings_interface)
        return fields

    def updateWidgets(self, prefix=None):
        super(ExternalVideoParagraphPropertiesEditForm, self).updateWidgets(prefix)
        if 'provider_name' in self.widgets:
            self.widgets['provider_name'].mode = DISPLAY_MODE

    def updateGroups(self):
        if 'provider_name' in self.widgets:
            provider = self.context.get_provider()
            if provider is not None:
                if 'integration_code' in self.widgets:  # custom video provider
                    self.add_group(NamedWidgetsGroup(self, 'integration_code', self.widgets, ('integration_code',),
                                                     bordered=False))
                elif 'video_id' in self.widgets:
                    self.add_group(NamedWidgetsGroup(self, 'video_id', self.widgets, ('video_id',), bordered=False))
                field_names = getFieldNamesInOrder(provider.settings_interface)
                if len(field_names) > 1:
                    self.add_group(NamedWidgetsGroup(self, 'provider_group', self.widgets,
                                                     field_names[1:],
                                                     bordered=False,
                                                     legend=_("Video provider settings"),
                                                     fieldset_class='margin-top-10 padding-y-5',
                                                     css_class='inner switcher padding-right-10 no-y-padding pull-left',
                                                     switch=True,
                                                     display_mode='never'))

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if 'title' in changes.get(IExternalVideoParagraph, ()):
            output.setdefault('events', []).append(
                get_json_paragraph_refresh_event(self.context, self.request))
        return output


@adapter_config(context=(IExternalVideoParagraph, IPyAMSLayer),
                provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json',
             context=IExternalVideoParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm, IPropertiesEditForm)
class ExternalVideoParagraphInnerEditForm(ExternalVideoParagraphPropertiesEditForm):
    """External video paragraph properties inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()
