#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

# import standard library
import re

from persistent import Persistent

# import interfaces
from pyams_content.component.video.interfaces import IExternalVideo, IExternalVideoProvider, IExternalVideoSettings
from pyams_content.component.video.provider.interfaces import IDailymotionVideoSettings
from pyams_content.features.checker.interfaces import IContentChecker

# import packages
from pyams_content.component.video import external_video_settings_factory
from pyams_content.features.checker import BaseContentChecker
from pyams_utils.adapter import adapter_config
from pyams_utils.registry import utility_config
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from pyams_content import _


DAILYMOTION_BASE_URL = re.compile(r'https?://(?:dai\.ly|www\.dailymotion\.com/video)/([^/]+)')


@implementer(IDailymotionVideoSettings)
class DailymotionVideoSettings(Persistent):
    """Dailymotion video settings"""

    _video_id = FieldProperty(IDailymotionVideoSettings['video_id'])
    start_at = FieldProperty(IDailymotionVideoSettings['start_at'])
    autoplay = FieldProperty(IDailymotionVideoSettings['autoplay'])
    show_info = FieldProperty(IDailymotionVideoSettings['show_info'])
    show_commands = FieldProperty(IDailymotionVideoSettings['show_commands'])
    ui_theme = FieldProperty(IDailymotionVideoSettings['ui_theme'])
    show_branding = FieldProperty(IDailymotionVideoSettings['show_branding'])
    show_endscreen = FieldProperty(IDailymotionVideoSettings['show_endscreen'])
    allow_fullscreen = FieldProperty(IDailymotionVideoSettings['allow_fullscreen'])
    allow_sharing = FieldProperty(IDailymotionVideoSettings['allow_sharing'])
    width = FieldProperty(IDailymotionVideoSettings['width'])
    height = FieldProperty(IDailymotionVideoSettings['height'])

    @property
    def video_id(self):
        return self._video_id

    @video_id.setter
    def video_id(self, value):
        if value:
            match = DAILYMOTION_BASE_URL.match(value)
            if match:
                value = match.groups()[0]
        self._video_id = value


@utility_config(name='Dailymotion', provides=IExternalVideoProvider)
class DailymotionVideoProvider(object):
    """Dailymotion video provider"""

    settings_interface = IDailymotionVideoSettings


@adapter_config(context=IExternalVideo, provides=IDailymotionVideoSettings)
def dailymotion_video_settings_factory(context):
    """Dailymotion video settings factory"""
    if context.provider_name != 'Dailymotion':
        return None
    return external_video_settings_factory(context)


@adapter_config(context=DailymotionVideoProvider, provides=IExternalVideoSettings)
def dailymotion_video_provider_settings_factory(context):
    """Dailymotion video provider settings factory"""
    return DailymotionVideoSettings()


@adapter_config(context=IDailymotionVideoSettings, provides=IContentChecker)
class DailymotionVideoSettingsContentChecker(BaseContentChecker):
    """Dailymotion video settings content checker"""

    label = _("Dailymotion settings")

    def inner_check(self, request):
        return []
