#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from z3c.form import button
from z3c.form.interfaces import INPUT_MODE
from zope.interface import implementer

from pyams_content.component.association.zmi.interfaces import IAssociationsParentForm
from pyams_content.component.paragraph import IBaseParagraph, IParagraphTitle
from pyams_content.component.paragraph.interfaces import IParagraphContainerTarget
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, \
    BaseParagraphAJAXEditForm, BaseParagraphAddForm, BaseParagraphAddMenu, \
    BaseParagraphPropertiesEditForm, IParagraphInnerEditFormButtons, \
    get_json_paragraph_refresh_event
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerView, \
    IParagraphInnerEditor
from pyams_content.component.verbatim.interfaces import IVerbatimParagraph, \
    VERBATIM_PARAGRAPH_TYPE
from pyams_content.component.verbatim.paragraph import VerbatimParagraph
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IInnerForm
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.text import get_text_start
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.interfaces import IPropertiesEditForm


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Verbatim paragraph
#

@viewlet_config(name='add-verbatim-paragraph.menu',
                context=IParagraphContainerTarget, layer=IPyAMSLayer,
                view=IParagraphContainerView, manager=IToolbarAddingMenu, weight=600)
class VerbatimParagraphAddMenu(BaseParagraphAddMenu):
    """Verbatim paragraph add menu"""

    label = _("Verbatim...")
    label_css_class = 'fa fa-fw ' + VerbatimParagraph.icon_class
    url = 'add-verbatim-paragraph.html'
    paragraph_type = VERBATIM_PARAGRAPH_TYPE


@pagelet_config(name='add-verbatim-paragraph.html',
                context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-verbatim-paragraph.json',
             context=IParagraphContainerTarget, layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class VerbatimParagraphAddForm(BaseParagraphAddForm):
    """Verbatim paragraph add form"""

    legend = _("Add new verbatim paragraph")

    content_interface = IVerbatimParagraph

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        if 'quote' in self.widgets:
            self.widgets['quote'].widget_css_class = 'input height-200'


@adapter_config(context=(IVerbatimParagraph, IPyAMSLayer),
                provides=IParagraphTitle)
def verbatim_paragraph_title_adapter(context, request):
    """Verbatim paragraph title adapter"""
    i18n = II18n(context)
    title = i18n.query_attribute('title', request=request)
    if not title:
        quote = i18n.query_attribute('quote', request=request)
        if quote:
            title = '<i>{0}</i>'.format(get_text_start(quote, 50, 10))
    return title


@pagelet_config(name='properties.html',
                context=IVerbatimParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json',
             context=IVerbatimParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
class VerbatimParagraphPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Verbatim paragraph properties edit form"""

    prefix = 'verbatim_properties.'

    legend = _("Edit verbatim paragraph properties")

    content_interface = IVerbatimParagraph

    def updateWidgets(self, prefix=None):
        super(VerbatimParagraphPropertiesEditForm, self).updateWidgets(prefix)
        if 'quote' in self.widgets:
            self.widgets['quote'].widget_css_class = 'input height-200'

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if 'quote' in changes.get(IVerbatimParagraph, ()):
            if 'title' not in changes.get(IBaseParagraph, ()):
                output.setdefault('events', []).append(
                    get_json_paragraph_refresh_event(self.context, self.request))
        return output


@adapter_config(context=(IVerbatimParagraph, IPyAMSLayer),
                provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json',
             context=IVerbatimParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm, IPropertiesEditForm, IAssociationsParentForm)
class VerbatimParagraphInnerEditForm(VerbatimParagraphPropertiesEditForm):
    """Verbatim paragraph inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        return button.Buttons()
