#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json

from pyramid.decorator import reify
from pyramid.view import view_config
from z3c.form import field
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import Interface, implementer

from pyams_content.component.verbatim import IVerbatim, IVerbatimContainer, \
    IVerbatimContainerTarget, Verbatim
from pyams_content.component.verbatim.interfaces import VERBATIM_HIDDEN_FIELDS
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.interfaces.form import IInnerSubForm
from pyams_form.security import ProtectedFormObjectMixin
from pyams_i18n.column import I18nAttrColumn
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.container import switch_element_visibility
from pyams_skin.event import get_json_switched_table_refresh_event, \
    get_json_table_row_refresh_event
from pyams_skin.interfaces.viewlet import IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import BaseTable, I18nColumn, SorterColumn, TrashColumn, \
    VisibilitySwitcherColumn
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.text import get_text_start
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from pyams_zmi.zmi.table import InnerTableView


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Views marker interfaces
#

class IVerbatimsView(Interface):
    """Verbatims view marker interface"""


class IVerbatimsParentForm(Interface):
    """Verbatims parent form marker interface"""


#
# Verbatim items table view
#

class VerbatimsTable(ProtectedFormObjectMixin, BaseTable):
    """Verbatims view inner table"""

    prefix = 'verbatims'

    hide_header = True
    hide_body_toolbar = True
    sortOn = None

    @classmethod
    def get_context(cls, context):
        return IVerbatimContainer(context)

    @reify
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover', 'table-tight']
        permission = self.permission
        if (not permission) or self.request.has_permission(permission, context=self.context):
            classes.append('table-dnd')
        return {'table': ' '.join(classes)}

    @reify
    def data_attributes(self):
        container = IVerbatimContainer(self.context)
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-location': absolute_url(container, self.request),
            'data-ams-tablednd-drag-handle': 'td.sorter',
            'data-ams-tablednd-drop-target': 'set-verbatims-order.json'
        })
        attributes.setdefault('td', {}).update({
            'data-ams-attribute-switcher': self.get_switcher_target,
            'data-ams-switcher-attribute-name': self.get_switcher_attribute
        })
        return attributes

    @staticmethod
    def get_switcher_target(element, column):
        if column.__name__ == 'show-hide':
            return 'switch-verbatim-visibility.json'

    @staticmethod
    def get_switcher_attribute(element, column):
        if column.__name__ == 'show-hide':
            return 'visible'

    @reify
    def values(self):
        return list(super().values)


@adapter_config(context=(IVerbatimContainer, IPyAMSLayer, VerbatimsTable), 
                provides=IValues)
class VerbatimsTableValuesAdapter(ContextRequestViewAdapter):
    """Verbatims table values adapter"""

    @property
    def values(self):
        return IVerbatimContainer(self.context).values()


@adapter_config(name='sorter', 
                context=(IVerbatimContainer, IPyAMSLayer, VerbatimsTable), 
                provides=IColumn)
class VerbatimsTableSorterColumn(ProtectedFormObjectMixin, SorterColumn):
    """Verbatims table sorter column"""


@view_config(name='set-verbatims-order.json', 
             context=IVerbatimContainer, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, renderer='json', xhr=True)
def set_verbatims_order(request):
    """Update verbatims order"""
    order = list(map(str, json.loads(request.params.get('names'))))
    request.context.updateOrder(order)
    return {'status': 'success'}


@adapter_config(name='show-hide', 
                context=(IVerbatimContainer, IPyAMSLayer, VerbatimsTable),
                provides=IColumn)
class VerbatimsTableShowHideColumn(ProtectedFormObjectMixin, VisibilitySwitcherColumn):
    """Verbatims container visibility switcher column"""


@view_config(name='switch-verbatim-visibility.json', 
             context=IVerbatimContainer, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, renderer='json', xhr=True)
def switch_verbatim_visibility(request):
    """Switch verbatim visibility"""
    return switch_element_visibility(request, IVerbatimContainer)


@adapter_config(name='quote',
                context=(IVerbatimContainer, IPyAMSLayer, VerbatimsTable), 
                provides=IColumn)
class VerbatimsTableQuoteColumn(I18nColumn, I18nAttrColumn):
    """Verbatims table quote column"""

    _header = _('verbatim-label', default="Header")
    attrName = 'quote'
    weight = 10

    def getValue(self, obj):
        return get_text_start(super().getValue(obj), 50) or '--'


@adapter_config(name='author',
                context=(IVerbatimContainer, IPyAMSLayer, VerbatimsTable), 
                provides=IColumn)
class VerbatimsTableAuthorColumn(I18nColumn, GetAttrColumn):
    """Verbatims table author column"""

    _header = _("Author")
    attrName = 'author'
    weight = 20

    def getValue(self, obj):
        return super().getValue(obj) or '--'


@adapter_config(name='trash',
                context=(IVerbatimContainer, IPyAMSLayer, VerbatimsTable),
                provides=IColumn)
class VerbatimsTableTrashColumn(ProtectedFormObjectMixin, TrashColumn):
    """Verbatims table trash column"""


@adapter_config(name='verbatims', 
                context=(IVerbatimContainerTarget, IPyAMSLayer, IVerbatimsParentForm),
                provides=IInnerSubForm)
@implementer(IVerbatimsView)
class VerbatimsView(InnerTableView):
    """Verbatims view"""

    title = _("Verbatims")

    table_class = VerbatimsTable
    weight = 110

    @property
    def actions_context(self):  # define context for internal actions
        return IVerbatimContainer(self.context)


#
# Verbatims forms
#

@viewlet_config(name='add-verbatim.action',
                context=IVerbatimContainerTarget, layer=IPyAMSLayer, view=IVerbatimsView,
                manager=IWidgetTitleViewletManager, weight=1)
class VerbatimAddAction(ProtectedFormObjectMixin, ToolbarAction):
    """Verbatim add action"""

    label = _("Add verbatim")
    label_css_class = 'fa fa-fw fa-plus'
    url = 'add-verbatim.html'
    modal_target = True


@pagelet_config(name='add-verbatim.html',
                context=IVerbatimContainer, layer=IPyAMSLayer)
@ajax_config(name='add-verbatim.json',
             context=IVerbatimContainer, layer=IPyAMSLayer,
             base=AJAXAddForm)
class VerbatimAddForm(AdminDialogAddForm):
    """Verbatim add form"""

    legend = _("Add new verbatim")
    icon_css_class = 'fa fa-fw fa-dashboard'

    fields = field.Fields(IVerbatim).omit(*VERBATIM_HIDDEN_FIELDS)
    edit_permission = None  # use context permission checker

    def create(self, data):
        return Verbatim()

    def add(self, object):
        IVerbatimContainer(self.context).append(object)

    def get_ajax_output(self, changes):
        return {
            'status': 'success',
            'message': self.request.localizer.translate(_("Verbatim was correctly added")),
            'events': [
                get_json_switched_table_refresh_event(self.context, self.request, VerbatimsTable)
            ]
        }


@pagelet_config(name='properties.html',
                context=IVerbatim, layer=IPyAMSLayer)
@ajax_config(name='properties.json',
             context=IVerbatim, layer=IPyAMSLayer)
class VerbatimPropertiesEditForm(AdminDialogEditForm):
    """Verbatim properties edit form"""

    prefix = 'verbatim_properties.'

    legend = _("Edit verbatim properties")
    icon_css_class = 'fa fa-fw fa-dashboard'

    fields = field.Fields(IVerbatim).omit(*VERBATIM_HIDDEN_FIELDS)
    edit_permission = None  # use context permission checker

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        updated = changes.get(IVerbatim, ())
        if updated:
            target = get_parent(self.context, IVerbatimContainerTarget)
            output.setdefault('events', []).append(
                get_json_table_row_refresh_event(target, self.request, VerbatimsTable,
                                                 self.context))
        return output
