#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_content.component.verbatim module

"""

__docformat__ = 'restructuredtext'

from persistent import Persistent
from pyramid.events import subscriber
from zope.container.contained import Contained
from zope.container.ordered import OrderedContainer
from zope.interface import implementer
from zope.lifecycleevent import IObjectAddedEvent, IObjectModifiedEvent, IObjectRemovedEvent, \
    ObjectModifiedEvent
from zope.location import locate
from zope.location.interfaces import ISublocations
from zope.schema.fieldproperty import FieldProperty
from zope.traversing.interfaces import ITraversable

from pyams_catalog.utils import index_object
from pyams_content.component.verbatim.interfaces import IVerbatim, IVerbatimContainer, \
    IVerbatimContainerTarget, VERBATIM_CONTAINER_KEY
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_LANG_VALUE, \
    MISSING_VALUE
from pyams_file.property import FileProperty
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_current_registry, get_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent

from pyams_content import _


#
# Verbatim class and adapters
#

@implementer(IVerbatim)
class Verbatim(Persistent, Contained):
    """Verbatim persistent class"""

    visible = FieldProperty(IVerbatim['visible'])
    quote = FieldProperty(IVerbatim['quote'])
    author = FieldProperty(IVerbatim['author'])
    charge = FieldProperty(IVerbatim['charge'])
    illustration = FileProperty(IVerbatim['illustration'])


@adapter_config(context=IVerbatim, provides=IFormContextPermissionChecker)
class VerbatimPermissionChecker(ContextAdapter):
    """Verbatim permission checker"""

    @property
    def edit_permission(self):
        content = get_parent(self.context, IVerbatimContainerTarget)
        return IFormContextPermissionChecker(content).edit_permission


@subscriber(IObjectAddedEvent, context_selector=IVerbatim)
def handle_added_verbatim(event):
    """Handle added verbatim"""
    content = get_parent(event.object, IVerbatimContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectModifiedEvent, context_selector=IVerbatim)
def handle_modified_verbatim(event):
    """Handle modified verbatim"""
    content = get_parent(event.object, IVerbatimContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectRemovedEvent, context_selector=IVerbatim)
def handle_removed_verbatim(event):
    """Handle removed verbatim"""
    content = get_parent(event.object, IVerbatimContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@adapter_config(context=IVerbatim, provides=IContentChecker)
class VerbatimContentChecker(BaseContentChecker):
    """Verbatim content checker"""

    @property
    def label(self):
        request = check_request()
        return II18n(self.context).query_attribute('title', request=request)

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language, )
        i18n = II18n(self.context)
        for lang in langs:
            for attr in ('quote', 'charge'):
                value = i18n.get_attribute(attr, lang, request)
                if not value:
                    field_title = translate(IVerbatim[attr].title)
                    if len(langs) == 1:
                        output.append(translate(MISSING_VALUE).format(field=field_title))
                    else:
                        output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        return output


#
# Verbatims container classes and adapters
#

@factory_config(IVerbatimContainer)
class VerbatimContainer(OrderedContainer):
    """Verbatim container"""

    last_id = 1

    def append(self, value, notify=True):
        key = str(self.last_id)
        if not notify:
            # pre-locate key number item to avoid multiple notifications
            locate(value, self, key)
        self[key] = value
        self.last_id += 1
        if not notify:
            # make sure that key number item is correctly indexed
            index_object(value)

    def get_visible_items(self):
        yield from filter(lambda x: IVerbatim(x).visible, self.values())


@adapter_config(context=IVerbatimContainerTarget, provides=IVerbatimContainer)
def verbatim_container_factory(target):
    """Verbatims container factory"""
    return get_annotation_adapter(target, VERBATIM_CONTAINER_KEY, IVerbatimContainer,
                                  name='++verbatims++')


@adapter_config(context=IVerbatimContainer, provides=IFormContextPermissionChecker)
class VerbatimContainerPermissionChecker(ContextAdapter):
    """Verbatims container permission checker"""

    @property
    def edit_permission(self):
        content = get_parent(self.context, IVerbatimContainerTarget)
        return IFormContextPermissionChecker(content).edit_permission


@adapter_config(name='verbatims', context=IVerbatimContainerTarget, provides=ITraversable)
class VerbatimContainerNamespace(ContextAdapter):
    """Verbatims container ++keynumbers++ namespace"""

    def traverse(self, name, furtherpath=None):
        return IVerbatimContainer(self.context)


@adapter_config(name='verbatims', context=IVerbatimContainerTarget, provides=ISublocations)
class VerbatimContainerSublocations(ContextAdapter):
    """Verbatims container sub-locations adapter"""

    def sublocations(self):
        return IVerbatimContainer(self.context).values()


@adapter_config(name='verbatims', context=IVerbatimContainerTarget, provides=IContentChecker)
class VerbatimContainerContentChecker(BaseContentChecker):
    """Verbatim container content checker"""

    label = _("Verbatims")
    sep = '\n'
    weight = 200

    def inner_check(self, request):
        output = []
        registry = request.registry
        for verbatim in IVerbatimContainer(self.context).values():
            if not verbatim.visible:
                continue
            for name, checker in sorted(registry.getAdapters((verbatim, ), IContentChecker),
                                        key=lambda x: x[1].weight):
                output.append('- {0} ({1}):'.format(verbatim.number,
                                                    II18n(verbatim).query_attribute('label', request=request) or '--'))
                output.append(checker.get_check_output(request))
        return output
