#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from pyams_content.component.theme import ICollectionsInfo, ICollectionsTarget, ITagsInfo, ITagsTarget, IThemesInfo, \
    IThemesTarget
from pyams_content.features.json import IJSONConverter, JSONBaseConverter
from pyams_content.reference.pictograms import IPictogramTable
from pyams_content.reference.pictograms.interfaces import IThesaurusTermPictogramsInfo
from pyams_i18n.interfaces import II18n
from pyams_skin.layer import IPyAMSLayer
from pyams_thesaurus.interfaces.term import IThesaurusTerm, STATUS_PUBLISHED
from pyams_utils.adapter import adapter_config
from pyams_utils.registry import query_utility
from pyams_utils.url import absolute_url


@adapter_config(required=(IThesaurusTerm, IPyAMSLayer),
                provides=IJSONConverter)
class JSONThesaurusTermConverter(JSONBaseConverter):
    """JSON thesaurus term converter"""

    def to_json(self, params=None):
        term = self.context
        result = {
            'name': term.label,
            'label': term.public_title
        }
        if term.definition:
            result['definition'] = term.definition
        if term.note:
            result['note'] = term.note
        pictograms_table = query_utility(IPictogramTable)
        if pictograms_table is not None:
            pictograms = {}
            pictograms_info = IThesaurusTermPictogramsInfo(term, None)
            if pictograms_info is not None:
                for mode in ('on', 'off'):
                    pictogram_name = getattr(pictograms_info, 'pictogram_{}'.format(mode))
                    if not pictogram_name:
                        continue
                    pictogram = pictograms_table.get(pictogram_name)
                    if pictogram is None:
                        continue
                    image = II18n(pictogram).query_attribute('image', request=self.request)
                    if not image:
                        continue
                    pictograms[mode] = {
                        'src': absolute_url(image, self.request),
                        'content_type': image.content_type
                    }
            if pictograms:
                result['pictograms'] = pictograms
        return result


@adapter_config(name='tags',
                required=(ITagsTarget, IPyAMSLayer),
                provides=IJSONConverter)
class JSONTagsConverter(JSONBaseConverter):
    """JSON tags converter"""

    is_inner = True

    @property
    def conversion_target(self):
        return None

    def convert_content(self, params):
        """Convert content to JSON"""
        result = super().convert_content(params)
        tags = ITagsInfo(self.context)
        if not tags.tags:
            return result
        result['items'] = [
            tag.public_title
            for tag in tags.tags
            if tag.status == STATUS_PUBLISHED
        ]
        return result


@adapter_config(name='themes',
                required=(IThemesTarget, IPyAMSLayer),
                provides=IJSONConverter)
class JSONThemesConverter(JSONBaseConverter):
    """JSON themes converter"""

    is_inner = True

    @property
    def conversion_target(self):
        return None

    def convert_content(self, params):
        """Convert content to JSON"""
        result = super().convert_content(params)
        themes = IThemesInfo(self.context)
        if not themes.themes:
            return result
        result['items'] = [
            theme.public_title
            for theme in themes.themes
            if theme.status == STATUS_PUBLISHED
        ]
        return result


@adapter_config(name='collections',
                required=(ICollectionsTarget, IPyAMSLayer),
                provides=IJSONConverter)
class JSONCollectionsConverter(JSONBaseConverter):
    """JSON collections converter"""

    is_inner = True

    @property
    def conversion_target(self):
        return None

    def convert_content(self, params):
        """Convert content to JSON"""
        result = super().convert_content(params)
        collections = ICollectionsInfo(self.context)
        if not collections.collections:
            return result
        result['items'] = [
            collection.public_title
            for collection in collections.collections
            if collection.status == STATUS_PUBLISHED
        ]
        return result
