#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from pyramid.events import subscriber
from zope.container.contained import Contained
from zope.interface import alsoProvides
from zope.lifecycleevent import IObjectAddedEvent
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.theme.interfaces import COLLECTIONS_INFO_KEY, \
    COLLECTIONS_MANAGER_KEY, ICollectionsInfo, ICollectionsManager, ICollectionsManagerTarget, \
    ICollectionsTarget, ITagsInfo, ITagsManager, ITagsManagerTarget, ITagsTarget, IThemesInfo, \
    IThemesManager, IThemesManagerTarget, IThemesTarget, TAGS_INFO_KEY, TAGS_MANAGER_KEY, \
    THEMES_INFO_KEY, THEMES_MANAGER_KEY
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.checker.interfaces import ERROR_VALUE, IContentChecker
from pyams_content.shared.site.interfaces import ISiteFolder
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.inherit import BaseInheritInfo, InheritedFieldProperty


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Tags management
#

@factory_config(ITagsManager)
class TagsManager(Persistent, Contained):
    """Tags manager persistent class"""

    thesaurus_name = FieldProperty(ITagsManager['thesaurus_name'])
    extract_name = FieldProperty(ITagsManager['extract_name'])

    enable_glossary = FieldProperty(ITagsManager['enable_glossary'])
    glossary_thesaurus_name = FieldProperty(ITagsManager['glossary_thesaurus_name'])


@adapter_config(required=ITagsManagerTarget,
                provides=ITagsManager)
def tags_manager_factory(target):
    """Tags manager factory"""
    return get_annotation_adapter(target, TAGS_MANAGER_KEY, ITagsManager,
                                  name='++tags-manager++')


@factory_config(ITagsInfo)
class TagsInfo(Persistent, Contained):
    """Tags info persistent class"""

    tags = FieldProperty(ITagsInfo['tags'])


@adapter_config(required=ITagsTarget,
                provides=ITagsInfo)
def tags_info_factory(target):
    """Tags info factory"""
    return get_annotation_adapter(target, TAGS_INFO_KEY, ITagsInfo,
                                  name='++tags++')


@adapter_config(name='tags',
                required=ITagsTarget,
                provides=IContentChecker)
class TagsContentChecker(BaseContentChecker):
    """Tags info content checker"""

    label = _("Tags")
    weight = 205

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        tags = ITagsInfo(self.context)
        if not tags.tags:
            output.append(translate(ERROR_VALUE).format(
                field=translate(ITagsInfo['tags'].title),
                message=translate(_("no defined tag"))))
        return output


#
# Themes management
#

@factory_config(IThemesManager)
class ThemesManager(Persistent, Contained):
    """Themes manager persistent class"""

    thesaurus_name = FieldProperty(IThemesManager['thesaurus_name'])
    extract_name = FieldProperty(IThemesManager['extract_name'])


@adapter_config(required=IThemesManagerTarget,
                provides=IThemesManager)
def themes_manager_factory(target):
    """Themes manager factory"""
    return get_annotation_adapter(target, THEMES_MANAGER_KEY, IThemesManager,
                                  name='++themes-manager++')


@factory_config(IThemesInfo)
class ThemesInfo(BaseInheritInfo, Persistent, Contained):
    """Themes info persistent class"""

    adapted_interface = IThemesInfo
    target_interface = IThemesTarget

    _themes = FieldProperty(IThemesInfo['themes'])
    themes = InheritedFieldProperty(IThemesInfo['themes'])


@adapter_config(required=IThemesTarget,
                provides=IThemesInfo)
def themes_info_factory(target):
    """Themes info factory"""
    return get_annotation_adapter(target, THEMES_INFO_KEY, IThemesInfo,
                                  name='++themes++')


@adapter_config(name='themes',
                required=IThemesTarget,
                provides=IContentChecker)
class ThemesContentChecker(BaseContentChecker):
    """Themes info content checker"""

    label = _("Themes")
    weight = 210

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        themes = IThemesInfo(self.context)
        if not themes.themes:
            output.append(translate(ERROR_VALUE).format(
                field=translate(IThemesInfo['themes'].title),
                message=translate(_("no defined theme"))))
        return output


@subscriber(IObjectAddedEvent,
            context_selector=ISiteFolder,
            parent_selector=IThemesTarget)
def handle_added_site_folder(event):
    """Handle site folder when added to a themes target"""
    alsoProvides(event.object, IThemesTarget)


#
# Collections management
#

@factory_config(ICollectionsManager)
class CollectionsManager(Persistent, Contained):
    """Collections manager persistent class"""

    thesaurus_name = FieldProperty(ICollectionsManager['thesaurus_name'])
    extract_name = FieldProperty(ICollectionsManager['extract_name'])


@adapter_config(required=ICollectionsManagerTarget,
                provides=ICollectionsManager)
def collections_manager_factory(target):
    """Collections manager factory"""
    return get_annotation_adapter(target, COLLECTIONS_MANAGER_KEY, ICollectionsManager,
                                  name='++collections-manager++')


@factory_config(ICollectionsInfo)
class CollectionsInfo(Persistent, Contained):
    """Collections info persistent class"""

    collections = FieldProperty(ICollectionsInfo['collections'])


@adapter_config(required=ICollectionsTarget,
                provides=ICollectionsInfo)
def collections_info_factory(target):
    """Collections info factory"""
    return get_annotation_adapter(target, COLLECTIONS_INFO_KEY, ICollectionsInfo,
                                  name='++collections++')


@adapter_config(name='collections',
                required=ICollectionsTarget,
                provides=IContentChecker)
class CollectionsContentChecker(BaseContentChecker):
    """Collections info content checker"""

    label = _("Collections")
    weight = 215

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        collections = ICollectionsInfo(self.context)
        if not collections.collections:
            output.append(translate(ERROR_VALUE).format(
                field=translate(ICollectionsInfo['collections'].title),
                message=translate(_("no defined collection"))))
        return output


