#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json

from pyramid.decorator import reify
from pyramid.view import view_config
from z3c.form import button, field
from z3c.form.interfaces import INPUT_MODE
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import Interface, implementer
from zope.schema.vocabulary import getVocabularyRegistry

from pyams_content.component.paragraph.interfaces import CONTENT_PARAGRAPHS_VOCABULARY, \
    IParagraphContainerTarget
from pyams_content.component.paragraph.interfaces.milestone import IMilestone, IMilestoneContainer, \
    IMilestoneContainerTarget, IMilestoneParagraph, MILESTONE_PARAGRAPH_TYPE
from pyams_content.component.paragraph.milestone import Milestone, MilestoneParagraph
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, \
    BaseParagraphAJAXEditForm, BaseParagraphAddForm, BaseParagraphAddMenu, \
    BaseParagraphPropertiesEditForm, IParagraphInnerEditFormButtons
from pyams_content.component.paragraph.zmi import IParagraphContainerView
from pyams_content.component.paragraph.zmi.interfaces import IParagraphInnerEditor
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.reference.pictograms.zmi.widget import PictogramSelectFieldWidget
from pyams_content.shared.common import IWfSharedContent
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.interfaces.form import IInnerForm, IInnerSubForm
from pyams_form.security import ProtectedFormObjectMixin
from pyams_i18n.column import I18nAttrColumn
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.container import switch_element_visibility
from pyams_skin.event import get_json_switched_table_refresh_event, \
    get_json_table_row_refresh_event, get_json_widget_refresh_event
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu, IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import BaseTable, I18nColumn, SorterColumn, TrashColumn, \
    VisibilitySwitcherColumn
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from pyams_zmi.zmi.table import InnerTableView


__docformat__ = 'restructuredtext'

from pyams_content import _


class IMilestonesView(Interface):
    """Milestones view marker interface"""


class IMilestonesParentForm(Interface):
    """Milestones parent form marker interface"""


@viewlet_config(name='add-milestone-paragraph.menu', context=IParagraphContainerTarget,
                view=IParagraphContainerView, layer=IPyAMSLayer, manager=IToolbarAddingMenu,
                weight=600)
class MilestoneParagraphAddMenu(BaseParagraphAddMenu):
    """Milestone paragraph add menu"""

    label = _("Milestones...")
    label_css_class = 'fa fa-fw ' + MilestoneParagraph.icon_class
    url = 'add-milestone-paragraph.html'
    paragraph_type = MILESTONE_PARAGRAPH_TYPE


@pagelet_config(name='add-milestone-paragraph.html', context=IParagraphContainerTarget,
                layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-milestone-paragraph.json', context=IParagraphContainerTarget,
             layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class MilestoneParagraphAddForm(BaseParagraphAddForm):
    """Milestone paragraph add form"""

    legend = _("Add new milestone paragraph")

    content_interface = IMilestoneParagraph


@pagelet_config(name='properties.html', context=IMilestoneParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IMilestoneParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
class MilestoneParagraphPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Milestone paragraph properties edit form"""

    prefix = 'milestones_properties.'

    @property
    def title(self):
        content = get_parent(self.context, IWfSharedContent)
        return II18n(content).query_attribute('title', request=self.request)

    legend = _("Edit milestone paragraph properties")

    content_interface = IMilestoneParagraph

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        updated = changes.get(IMilestoneParagraph, ())
        if 'renderer' in updated:
            form = MilestoneParagraphInnerEditForm(self.context, self.request)
            form.update()
            output.setdefault('events', []) \
                .append(get_json_widget_refresh_event(self.context, self.request,
                                                      MilestoneParagraphInnerEditForm, 'renderer'))
        return output


@adapter_config(context=(IMilestoneParagraph, IPyAMSLayer),
                provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json', context=IMilestoneParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm, IMilestonesParentForm)
class MilestoneParagraphInnerEditForm(MilestoneParagraphPropertiesEditForm):
    """Milestone paragraph inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()


#
# Milestone items table view
#

class MilestonesTable(ProtectedFormObjectMixin, BaseTable):
    """Milestones view inner table"""

    prefix = 'milestones'

    hide_header = True
    sortOn = None

    @property
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover', 'table-tight']
        permission = self.permission
        if (not permission) or self.request.has_permission(permission, context=self.context):
            classes.append('table-dnd')
        return {'table': ' '.join(classes)}

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes['table'] = {
            'id': self.id,
            'data-ams-location': absolute_url(IMilestoneContainer(self.context), self.request),
            'data-ams-tablednd-drag-handle': 'td.sorter',
            'data-ams-tablednd-drop-target': 'set-milestones-order.json'
        }
        attributes.setdefault('td', {}).update({
            'data-ams-attribute-switcher': self.get_switcher_target
        })
        return attributes

    @staticmethod
    def get_switcher_target(element, column):
        if column.__name__ == 'show-hide':
            return 'switch-milestone-visibility.json'

    @reify
    def values(self):
        return list(super().values)


@adapter_config(context=(IMilestoneContainerTarget, IPyAMSLayer, MilestonesTable),
                provides=IValues)
class MilestonesTableValuesAdapter(ContextRequestViewAdapter):
    """Milestones table values adapter"""

    @property
    def values(self):
        return IMilestoneContainer(self.context).values()


@adapter_config(name='sorter',
                context=(IMilestoneContainerTarget, IPyAMSLayer, MilestonesTable),
                provides=IColumn)
class MilestonesTableSorterColumn(ProtectedFormObjectMixin, SorterColumn):
    """Milestones table sorter column"""


@view_config(name='set-milestones-order.json', context=IMilestoneContainer,
             request_type=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION,
             renderer='json', xhr=True)
def set_milestones_order(request):
    """Update milestones order"""
    order = list(map(str, json.loads(request.params.get('names'))))
    request.context.updateOrder(order)
    return {'status': 'success'}


@adapter_config(name='show-hide',
                context=(IMilestoneContainerTarget, IPyAMSLayer, MilestonesTable),
                provides=IColumn)
class MilestonesTableShowHideColumn(ProtectedFormObjectMixin, VisibilitySwitcherColumn):
    """Milestones container visibility switcher column"""


@view_config(name='switch-milestone-visibility.json', context=IMilestoneContainer,
             request_type=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION,
             renderer='json', xhr=True)
def switch_milestone_visibility(request):
    """Set milestone visibility"""
    return switch_element_visibility(request, IMilestoneContainer)


@adapter_config(name='name',
                context=(IMilestoneContainerTarget, IPyAMSLayer, MilestonesTable),
                provides=IColumn)
class MilestonesTableNameColumn(I18nColumn, I18nAttrColumn):
    """Milestones table name column"""

    _header = _("Title")
    attrName = 'title'
    weight = 10

    def getValue(self, obj):
        return super(MilestonesTableNameColumn, self).getValue(obj) or '--'


@adapter_config(name='info',
                context=(IMilestoneContainerTarget, IPyAMSLayer, MilestonesTable),
                provides=IColumn)
class MilestonesTableInfoColumn(I18nColumn, I18nAttrColumn):
    """Milestones table information column"""

    _header = _("Associated label")
    attrName = 'label'
    weight = 20

    def getValue(self, obj):
        return super(MilestonesTableInfoColumn, self).getValue(obj) or '--'


@adapter_config(name='anchor', context=(IMilestoneContainerTarget, IPyAMSLayer, MilestonesTable),
                provides=IColumn)
class MilestonesTableAnchorColumn(I18nColumn, GetAttrColumn):
    """Milestones table anchor column"""

    _header = _("Anchor")
    weight = 30

    defaultValue = ' '.join(('-',) * 8)

    def getValue(self, obj):
        if not obj.anchor:
            return self.defaultValue
        target = get_parent(self.context, IParagraphContainerTarget)
        if target is None:
            return self.defaultValue
        try:
            vocabulary = getVocabularyRegistry().get(obj, CONTENT_PARAGRAPHS_VOCABULARY)
            return vocabulary.getTermByToken(obj.anchor).title
        except LookupError:
            return self.request.localizer.translate(_("(missing paragraph)")).format(obj.anchor)


@adapter_config(name='trash', context=(IMilestoneContainerTarget, IPyAMSLayer, MilestonesTable),
                provides=IColumn)
class MilestonesTableTrashColumn(ProtectedFormObjectMixin, TrashColumn):
    """Milestones table trash column"""


@adapter_config(name='milestones',
                context=(IMilestoneContainerTarget, IPyAMSLayer, IMilestonesParentForm),
                provides=IInnerSubForm)
@implementer(IMilestonesView)
class MilestonesView(InnerTableView):
    """Milestones view"""

    title = _("Milestones")

    table_class = MilestonesTable
    weight = 100


#
# Milestones forms
#

@viewlet_config(name='add-milestone.action', context=IMilestoneContainerTarget, layer=IPyAMSLayer,
                view=IMilestonesView, manager=IWidgetTitleViewletManager,
                permission=MANAGE_CONTENT_PERMISSION, weight=1)
class MilestoneAddAction(ToolbarAction):
    """Milestone add action"""

    label = _("Add milestone")
    label_css_class = 'fa fa-fw fa-plus'
    url = 'add-milestone.html'
    modal_target = True


@pagelet_config(name='add-milestone.html', context=IMilestoneContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-milestone.json', context=IMilestoneContainerTarget, layer=IPyAMSLayer,
             base=AJAXAddForm)
class MilestoneAddForm(AdminDialogAddForm):
    """Milestone add form"""

    legend = _("Add new milestone")
    icon_css_class = 'fa fa-fw fa-arrow-h'

    fields = field.Fields(IMilestone).omit('__parent__', '__name__', 'visible')
    fields['pictogram_name'].widgetFactory = PictogramSelectFieldWidget

    edit_permission = MANAGE_CONTENT_PERMISSION

    def updateWidgets(self, prefix=None):
        super(MilestoneAddForm, self).updateWidgets(prefix)
        if 'label' in self.widgets:
            self.widgets['label'].widget_css_class = 'input height-100'

    def create(self, data):
        return Milestone()

    def add(self, object):
        IMilestoneContainer(self.context).append(object)

    def get_ajax_output(self, changes):
        return {
            'status': 'success',
            'message': self.request.localizer.translate(_("Milestone was correctly added")),
            'events': [
                get_json_switched_table_refresh_event(self.context, self.request, MilestonesTable)]
        }


@pagelet_config(name='properties.html', context=IMilestone, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IMilestone, layer=IPyAMSLayer)
class MilestonePropertiesEditForm(AdminDialogEditForm):
    """Milestone properties edit form"""

    prefix = 'milestone_properties.'

    legend = _("Edit milestone properties")
    icon_css_class = 'fa fa-fw fa-arrows-h'

    fields = field.Fields(IMilestone).omit('__parent__', '__name__', 'visible')
    fields['pictogram_name'].widgetFactory = PictogramSelectFieldWidget

    edit_permission = MANAGE_CONTENT_PERMISSION

    def updateWidgets(self, prefix=None):
        super(MilestonePropertiesEditForm, self).updateWidgets(prefix)
        if 'label' in self.widgets:
            self.widgets['label'].widget_css_class = 'input height-100'

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if changes:
            target = get_parent(self.context, IMilestoneContainerTarget)
            output.setdefault('events', []).append(
                get_json_table_row_refresh_event(target, self.request,
                                                 MilestonesTable, self.context))
        return output
