#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from z3c.form import button
from z3c.form.interfaces import INPUT_MODE
from zope.interface import alsoProvides, implementer

from pyams_content.component.association.interfaces import IAssociationContainerTarget
from pyams_content.component.association.zmi import AssociationsTable
from pyams_content.component.association.zmi.interfaces import IAssociationsParentForm
from pyams_content.component.paragraph.html import HTMLParagraph, RawParagraph
from pyams_content.component.paragraph.interfaces import IBaseParagraph, IParagraphContainerTarget, \
    IParagraphFactorySettings, IParagraphTitle
from pyams_content.component.paragraph.interfaces.html import HTML_PARAGRAPH_TYPE, IHTMLParagraph, \
    IRawParagraph, RAW_PARAGRAPH_TYPE
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, \
    BaseParagraphAJAXEditForm, BaseParagraphAddForm, BaseParagraphAddMenu, \
    BaseParagraphPropertiesEditForm, IParagraphInnerEditFormButtons, \
    get_json_paragraph_refresh_event, get_json_paragraph_toolbar_refresh_event
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerView, \
    IParagraphInnerEditor
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IInnerForm
from pyams_form.security import ProtectedFormObjectMixin
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_switched_table_refresh_event
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.menu import MenuDivider
from pyams_utils.adapter import adapter_config
from pyams_utils.html import html_to_text
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.text import get_text_start
from pyams_utils.traversing import get_parent
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.interfaces import IPropertiesEditForm


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Raw HTML paragraph
#

@viewlet_config(name='add-raw-paragraph.divider', context=IParagraphContainerTarget,
                view=IParagraphContainerView, layer=IPyAMSLayer, manager=IToolbarAddingMenu,
                weight=990)
class RawParagraphAddMenuDivider(ProtectedFormObjectMixin, MenuDivider):
    """Source code paragraph add menu divider"""

    def __new__(cls, context, request, view, manager):
        settings = get_parent(context, IParagraphFactorySettings)
        if (settings is not None) and (
                RAW_PARAGRAPH_TYPE not in (settings.allowed_paragraphs or ())):
            return None
        return MenuDivider.__new__(cls)


@viewlet_config(name='add-raw-paragraph.menu', context=IParagraphContainerTarget,
                view=IParagraphContainerView, layer=IPyAMSLayer, manager=IToolbarAddingMenu,
                weight=999)
class RawParagraphAddMenu(BaseParagraphAddMenu):
    """Source code paragraph add menu"""

    label = _("Source code...")
    label_css_class = 'fa fa-fw ' + RawParagraph.icon_class
    url = 'add-raw-paragraph.html'
    paragraph_type = RAW_PARAGRAPH_TYPE


@pagelet_config(name='add-raw-paragraph.html', context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-raw-paragraph.json', context=IParagraphContainerTarget, layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class RawParagraphAddForm(BaseParagraphAddForm):
    """Source code paragraph add form"""

    legend = _("Add new source code paragraph")
    dialog_class = 'modal-large'

    label_css_class = 'control-label col-md-2'
    input_css_class = 'col-md-10'

    content_interface = IRawParagraph

    def updateWidgets(self, prefix=None):
        super(RawParagraphAddForm, self).updateWidgets(prefix)
        if 'body' in self.widgets:
            widget = self.widgets['body']
            widget.widget_css_class = 'monospace input height-400 text-editor'
            widget.object_data = {
                'ams-editor-mode': 'ace/mode/html'
            }
            alsoProvides(widget, IObjectData)


@pagelet_config(name='properties.html', context=IRawParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IRawParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
class RawParagraphPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Source code paragraph properties edit form"""

    prefix = 'raw_properties.'

    legend = _("Edit source code paragraph properties")

    content_interface = IRawParagraph

    def updateWidgets(self, prefix=None):
        super(RawParagraphPropertiesEditForm, self).updateWidgets(prefix)
        if 'body' in self.widgets:
            widget = self.widgets['body']
            widget.widget_css_class = 'monospace input height-400 text-editor'
            widget.object_data = {
                'ams-editor-mode': 'ace/mode/html'
            }
            alsoProvides(widget, IObjectData)


@adapter_config(context=(IRawParagraph, IPyAMSLayer), provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json', context=IRawParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm)
class RawParagraphInnerEditForm(RawParagraphPropertiesEditForm):
    """Source code paragraph inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()


#
# Rich text paragraph
#

@viewlet_config(name='add-html-paragraph.menu', context=IParagraphContainerTarget,
                view=IParagraphContainerView, layer=IPyAMSLayer, manager=IToolbarAddingMenu,
                weight=50)
class HTMLParagraphAddMenu(BaseParagraphAddMenu):
    """Rich text paragraph add menu"""

    label = _("Rich text...")
    label_css_class = 'fa fa-fw ' + HTMLParagraph.icon_class
    url = 'add-html-paragraph.html'
    paragraph_type = HTML_PARAGRAPH_TYPE


@pagelet_config(name='add-html-paragraph.html', context=IParagraphContainerTarget,
                layer=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-html-paragraph.json', context=IParagraphContainerTarget, layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class HTMLParagraphAddForm(BaseParagraphAddForm):
    """Rich text paragraph add form"""

    legend = _("Add new rich text paragraph")
    dialog_class = 'modal-max'

    label_css_class = 'control-label col-md-2'
    input_css_class = 'col-md-10'

    content_interface = IHTMLParagraph


@adapter_config(context=(IHTMLParagraph, IPyAMSLayer), provides=IParagraphTitle)
def html_paragraph_title_adapter(context, request):
    """HTML paragraph title adapter"""
    i18n = II18n(context)
    title = i18n.query_attribute('title', request=request)
    if not title:
        body = i18n.query_attribute('body', request=request)
        if body:
            title = '<i>{0}</i>'.format(get_text_start(html_to_text(body), 50, 10))
    return title


@pagelet_config(name='properties.html', context=IHTMLParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IHTMLParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
class HTMLParagraphPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Rich text paragraph properties edit form"""

    prefix = 'html_properties.'

    legend = _("Edit rich text paragraph properties")
    dialog_class = 'modal-max'

    label_css_class = 'control-label col-md-2'
    input_css_class = 'col-md-10'

    content_interface = IHTMLParagraph

    def updateWidgets(self, prefix=None):
        super(HTMLParagraphPropertiesEditForm, self).updateWidgets(prefix)
        if 'body' in self.widgets:
            body_widget = self.widgets['body']
            for lang in body_widget.langs:
                widget = body_widget.widgets[lang]
                widget.id = '{id}_{name}'.format(id=widget.id, name=self.context.__name__)

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if 'body' in changes.get(IHTMLParagraph, ()):
            # refresh paragraph title
            if 'title' not in changes.get(IBaseParagraph, ()):
                output.setdefault('events', []).append(
                    get_json_paragraph_refresh_event(self.context, self.request))
            # refresh associations count markers
            parent = get_parent(self.context, IAssociationContainerTarget)
            output.setdefault('events', []).append(
                get_json_paragraph_toolbar_refresh_event(parent, self.request))
            # refresh associations table
            output.setdefault('events', []).append(
                get_json_switched_table_refresh_event(self.context, self.request,
                                                      AssociationsTable))
        return output


@adapter_config(context=(IHTMLParagraph, IPyAMSLayer), provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json', context=IHTMLParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm, IPropertiesEditForm, IAssociationsParentForm)
class HTMLParagraphInnerEditForm(HTMLParagraphPropertiesEditForm):
    """Rich text paragraph inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()
