#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from transaction.interfaces import ITransactionManager
from z3c.form import button
from z3c.form.interfaces import INPUT_MODE
from zope.interface import implementer

from pyams_content.component.association.zmi.interfaces import IAssociationsParentForm
from pyams_content.component.paragraph.audio import AudioParagraph
from pyams_content.component.paragraph.interfaces import IParagraphContainerTarget
from pyams_content.component.paragraph.interfaces.audio import AUDIO_PARAGRAPH_TYPE, IAudioParagraph
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, BaseParagraphAJAXEditForm, \
    BaseParagraphAddForm, BaseParagraphAddMenu, BaseParagraphPropertiesEditForm, IParagraphInnerEditFormButtons, \
    get_json_paragraph_refresh_event
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerView, IParagraphInnerEditor
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IInnerForm
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_form_refresh_event, get_json_widget_refresh_event
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.interfaces import IPropertiesEditForm

from pyams_content import _


@viewlet_config(name='add-audio-paragraph.menu', context=IParagraphContainerTarget, view=IParagraphContainerView,
                layer=IPyAMSLayer, manager=IToolbarAddingMenu, weight=80)
class AudioParagraphAddMenu(BaseParagraphAddMenu):
    """Audio paragraph add menu"""

    label = _("Audio paragraph...")
    label_css_class = 'fa fa-fw ' + AudioParagraph.icon_class
    url = 'add-audio-paragraph.html'
    paragraph_type = AUDIO_PARAGRAPH_TYPE


@pagelet_config(name='add-audio-paragraph.html', context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-audio-paragraph.json', context=IParagraphContainerTarget, layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class AudioParagraphAddForm(BaseParagraphAddForm):
    """Audio paragraph add form"""

    legend = _("Add new audio paragraph")
    dialog_class = 'modal-large'

    content_interface = IAudioParagraph


@pagelet_config(name='properties.html', context=IAudioParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IAudioParagraph, layer=IPyAMSLayer, base=BaseParagraphAJAXEditForm)
class AudioParagraphPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Audio paragraph properties edit form"""

    prefix = 'audio_properties.'

    legend = _("Edit audio properties")
    dialog_class = 'modal-large'

    content_interface = IAudioParagraph

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        updated = changes.get(IAudioParagraph, ())
        if 'data' in updated:
            # we have to commit transaction to be able to handle blobs...
            ITransactionManager(self.context).get().commit()
            output.setdefault('events', []).append(
                get_json_form_refresh_event(self.context, self.request, AudioParagraphPropertiesInnerEditForm))
        else:
            if 'title' in updated:
                output.setdefault('events', []).append(
                    get_json_paragraph_refresh_event(self.context, self.request))
            if 'renderer' in updated:
                output.setdefault('events', []).append(
                    get_json_widget_refresh_event(self.context, self.request,
                                                  AudioParagraphPropertiesInnerEditForm, 'renderer'))
        return output


@adapter_config(context=(IAudioParagraph, IPyAMSLayer), provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json', context=IAudioParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm, IPropertiesEditForm, IAssociationsParentForm)
class AudioParagraphPropertiesInnerEditForm(AudioParagraphPropertiesEditForm):
    """Audio paragraph properties inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()
