#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.decorator import reify
from pyramid.location import lineage
from z3c.form import button, field
from z3c.form.interfaces import DISPLAY_MODE
from zope.interface import Interface

from pyams_content.component.paragraph import IParagraphContainer, IParagraphTitle
from pyams_content.component.paragraph.interfaces import IBaseParagraph, IParagraphContainerTarget, \
    IParagraphFactory, IParagraphFactorySettings, IParagraphRenderer, PARAGRAPH_HIDDEN_FIELDS
from pyams_content.component.paragraph.zmi.container import ParagraphContainerBaseTable, \
    ParagraphContainerTable
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerView
from pyams_content.features.renderer.zmi import BaseRenderedContentRenderer
from pyams_content.features.renderer.zmi.widget import RendererFieldWidget
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION, MANAGE_TOOL_PERMISSION, \
    PUBLISH_CONTENT_PERMISSION
from pyams_content.shared.common.interfaces import IWfSharedContent
from pyams_content.shared.site.interfaces import ISiteManager
from pyams_form.form import AJAXAddForm, AJAXEditForm, ajax_config
from pyams_form.help import FormHelp
from pyams_form.interfaces.form import IFormHelp, check_submit_button
from pyams_form.schema import ActionButton, CloseButton, ResetButton
from pyams_form.security import ProtectedFormObjectMixin
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_switched_table_refresh_event, get_json_widget_refresh_event
from pyams_skin.interfaces.container import ITableElementName
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import get_element_id
from pyams_skin.viewlet.menu import MenuDivider, MenuItem
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import get_object_factory
from pyams_utils.registry import query_utility
from pyams_utils.traversing import get_parent
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Default paragraphs settings
#

@viewlet_config(name='default-paragraphs.menu', context=IParagraphFactorySettings,
                layer=IAdminLayer, manager=IPropertiesMenu, permission=MANAGE_TOOL_PERMISSION,
                weight=610)
class DefaultParagraphsSettingsMenu(MenuItem):
    """Default paragraphs settings menu"""

    label = _("Content block types...")
    icon_class = 'fa-paragraph'
    url = 'default-paragraphs.html'
    modal_target = True


@pagelet_config(name='default-paragraphs.html', context=IParagraphFactorySettings,
                layer=IPyAMSLayer, permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='default-paragraphs.json', context=IParagraphFactorySettings, layer=IPyAMSLayer)
class DefaultParagraphsEditForm(AdminDialogEditForm):
    """Default paragraphs edit form"""

    prefix = 'default_paragraphs.'

    legend = _("Content block types")

    fields = field.Fields(IParagraphFactorySettings)
    edit_permission = MANAGE_TOOL_PERMISSION


@adapter_config(context=(IParagraphFactorySettings, IPyAMSLayer, DefaultParagraphsEditForm),
                provides=IFormHelp)
class DefaultParagraphsEditFormHelp(FormHelp):
    """Default paragraphs edit form help"""

    message = _("You can define which types of paragraphs are allowed in this container.\n\n"
                "Default paragraphs will be added automatically (in selected order) to any new "
                "created content.\n\n"
                "NOTICE: removing types from allowed types list will have no effect on already "
                "created contents!")
    message_format = 'rest'


#
# Base paragraph forms
#

def get_json_paragraph_refresh_event(context, request):
    """Get JSON response value for paragraph refresh event"""
    parent = get_parent(context, IParagraphContainerTarget)
    if parent is not None:
        adapter = request.registry.queryMultiAdapter((context, request), IParagraphTitle)
        if adapter is not None:
            title = adapter
        else:
            title = II18n(context).query_attribute('title', request=request)
        return {
            'event': 'myams.refresh',
            'options': {
                'handler': 'PyAMS_content.paragraphs.refreshParagraph',
                'object_id': get_element_id(ParagraphContainerBaseTable, context, parent),
                'title': title,
                'visible': context.visible
            }
        }


def get_json_paragraph_toolbar_refresh_event(context, request, table_factory=None,
                                             viewlet_factory=None):
    """Get JSON response value for paragraph toolbar refresh event"""
    if table_factory is None:
        from pyams_content.component.paragraph.zmi.container \
            import ParagraphContainerTable as table_factory
    if viewlet_factory is None:
        from pyams_content.component.paragraph.zmi.container \
            import ParagraphTitleToolbarViewletManager as viewlet_factory

    parent = get_parent(context, IParagraphContainerTarget)
    if parent is not None:
        table = table_factory(context, request)
        viewlet = viewlet_factory(context, request, table)
        viewlet.update()
        return {
            'event': 'myams.refresh',
            'options': {
                'handler': 'PyAMS_content.paragraphs.updateToolbar',
                'object_id': get_element_id(ParagraphContainerBaseTable, context, parent),
                'toolbar_tag': viewlet.render()
            }
        }


def get_json_paragraph_markers_refresh_event(context, request, form, viewlet_factory,
                                             marker_type=None):
    """Get JSON response value for paragraph markers refresh event"""
    parent = get_parent(context, IParagraphContainerTarget)
    if parent is not None:
        marker = viewlet_factory(context, request, form, None)
        if marker is not None:
            marker.update()
            return {
                'event': 'myams.refresh',
                'options': {
                    'handler': 'PyAMS_content.paragraphs.updateMarkers',
                    'object_id': get_element_id(ParagraphContainerBaseTable, context, parent),
                    'marker_type': marker_type or marker.marker_type,
                    'marker_tag': marker.render()
                }
            }


class BaseParagraphAddMenu(ProtectedFormObjectMixin, ToolbarMenuItem):
    """Base paragraph add menu"""

    paragraph_type = None
    modal_target = True

    def __new__(cls, context, request, view, manager):
        settings = get_parent(context, IParagraphFactorySettings)
        if (settings is not None) and (
                cls.paragraph_type not in (settings.allowed_paragraphs or ())):
            return None
        return ToolbarMenuItem.__new__(cls)


@viewlet_config(name='custom-paragraphs.divider', context=IParagraphContainerTarget,
                view=IParagraphContainerView, layer=IPyAMSLayer, manager=IToolbarAddingMenu,
                weight=500)
class ParagraphsAddMenuDivider(ProtectedFormObjectMixin, MenuDivider):
    """Paragraphs add menu divider"""

    def __new__(cls, context, request, view, manager):
        settings = get_parent(context, IParagraphFactorySettings)
        if settings is None:
            return MenuDivider.__new__(cls)
        for factory_name in settings.allowed_paragraphs or ():
            factory = query_utility(IParagraphFactory, name=factory_name)
            if (factory is not None) and factory.secondary_menu:
                return MenuDivider.__new__(cls)
        return None


#
# Base paragraphs add forms
#

class BaseParagraphAddForm(AdminDialogAddForm):
    """Base paragraph add form"""

    content_interface = None

    @reify
    def factory(self):
        return get_object_factory(self.content_interface)

    @property
    def icon_css_class(self):
        return 'fa fa-fw {}'.format(self.factory.factory.icon_class)

    @property
    def fields(self):
        return field.Fields(self.content_interface).omit(*PARAGRAPH_HIDDEN_FIELDS)

    edit_permission = MANAGE_CONTENT_PERMISSION

    def create(self, data):
        return self.factory()

    def add(self, object):
        IParagraphContainer(self.context).append(object)


class BaseParagraphAJAXAddForm(AJAXAddForm):
    """Base paragraph AJAX add form"""

    def get_ajax_output(self, changes):
        if IWfSharedContent.providedBy(self.context) or ISiteManager.providedBy(self.context):
            table_factory = ParagraphContainerTable
        else:
            table_factory = ParagraphContainerBaseTable
        event = get_json_switched_table_refresh_event(self.context, self.request, table_factory)
        table_id = event['options']['object_id']
        return {
            'status': 'success',
            'message': self.request.localizer.translate(_("Paragraph was correctly added.")),
            'events': [event],
            'callbacks': [{
                'callback': 'PyAMS_content.paragraphs.switchLastEditor',
                'options': table_id
            }]
        }


#
# Base paragraph edit forms
#

class IParagraphEditFormButtons(Interface):
    """Paragraph edit form buttons"""

    preview = ActionButton(name='preview', title=_("Preview"),
                           label_css_class='fa fa-fw fa-binoculars',
                           url='content-preview.html',
                           modal_target=True)

    close = CloseButton(name='close', title=_("Cancel"))

    submit = button.Button(name='submit', title=_("Submit"), condition=check_submit_button)


class IParagraphInnerEditFormButtons(Interface):
    """Paragraph inner edit form buttons"""

    preview = ActionButton(name='preview', title=_("Preview"),
                           label_css_class='fa fa-fw fa-binoculars',
                           url='content-preview.html',
                           modal_target=True)

    cancel = ResetButton(name='cancel', title=_("Cancel"))

    submit = button.Button(name='submit', title=_("Submit"), condition=check_submit_button)


class BaseParagraphPropertiesEditForm(AdminDialogEditForm):
    """Base paragraph edit form"""

    prefix = 'paragraph.'
    buttons = button.Buttons(IParagraphEditFormButtons)

    content_interface = None

    @property
    def title(self):
        registry = self.request.registry
        for parent in lineage(self.context):
            adapter = registry.queryMultiAdapter((parent, self.request), ITableElementName)
            if adapter is not None:
                return adapter.name
        content = get_parent(self.context, IWfSharedContent)
        if content is not None:
            return II18n(content).query_attribute('title', request=self.request)
        return super(BaseParagraphPropertiesEditForm, self).title

    @reify
    def factory(self):
        return get_object_factory(self.content_interface)

    @property
    def icon_css_class(self):
        return 'fa fa-fw {}'.format(self.factory.factory.icon_class)

    @property
    def fields(self):
        fields = field.Fields(self.content_interface).omit(*PARAGRAPH_HIDDEN_FIELDS)
        if 'renderer' in fields:
            fields['renderer'].widgetFactory = RendererFieldWidget
        return fields

    edit_permission = MANAGE_CONTENT_PERMISSION

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        if self.context.locked:
            title = self.widgets.get('title')
            if (title is not None) and \
                    not self.request.has_permission(PUBLISH_CONTENT_PERMISSION,
                                                    context=self.context):
                title.set_mode(DISPLAY_MODE)


class BaseParagraphAJAXEditForm(AJAXEditForm):
    """Base paragraph AJAX edit form"""

    def get_ajax_output(self, changes):
        output = super(BaseParagraphAJAXEditForm, self).get_ajax_output(changes)
        if 'title' in changes.get(IBaseParagraph, ()):
            output.setdefault('events', []).append(
                get_json_paragraph_refresh_event(self.context, self.request))
        elif 'renderer' in self.widgets:
            renderer_interface = self.widgets['renderer'].field.interface
            if 'renderer' in changes.get(renderer_interface, ()):
                output.setdefault('events', []).append(
                    get_json_widget_refresh_event(self.context, self.request, self.__class__,
                                                  'renderer'))
                renderer = self.context.get_renderer(self.request)
                if (renderer is not None) and \
                        (renderer.settings_interface is not None):
                    output['smallbox'] = {
                        'status': 'info',
                        'message': self.request.localizer.translate(
                            _("You changed renderer selection. Don't omit to "
                              "update new renderer properties...")),
                        'timeout': 5000
                    }
        return output


#
# Base paragraph renderer
#

@adapter_config(context=(IBaseParagraph, IPyAMSLayer), provides=IParagraphRenderer)
class BaseParagraphRenderer(BaseRenderedContentRenderer):
    """Base paragraph renderer"""
