#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from pyramid.events import subscriber
from zope.container.contained import Contained
from zope.container.ordered import OrderedContainer
from zope.interface import implementer
from zope.lifecycleevent import IObjectAddedEvent, ObjectModifiedEvent
from zope.location import locate
from zope.location.interfaces import ISublocations
from zope.schema.fieldproperty import FieldProperty
from zope.traversing.interfaces import ITraversable

from pyams_catalog.utils import index_object
from pyams_content.component.illustration import IBasicIllustrationTarget
from pyams_content.component.paragraph import BaseParagraph, BaseParagraphContentChecker, BaseParagraphFactory
from pyams_content.component.paragraph import IParagraphFactory
from pyams_content.component.paragraph.api import JSONParagraphConverter
from pyams_content.component.paragraph.interfaces.pictogram import IPictogramContainer, IPictogramContainerTarget, \
    IPictogramItem, IPictogramParagraph, PICTOGRAM_CONTAINER_KEY, PICTOGRAM_PARAGRAPH_NAME, \
    PICTOGRAM_PARAGRAPH_RENDERERS, PICTOGRAM_PARAGRAPH_TYPE
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.checker.interfaces import ERROR_VALUE, IContentChecker, MISSING_LANG_VALUE, MISSING_VALUE
from pyams_content.features.json import IJSONConverter
from pyams_content.features.renderer import RenderersVocabulary
from pyams_content.reference.pictograms.interfaces import IPictogramTable
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_current_registry, get_utility, query_utility, utility_config
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config
from pyams_utils.zodb import volatile_property

from pyams_content import _


#
# Pictogram item class and adapters
#

@implementer(IPictogramItem)
class PictogramItem(Persistent, Contained):
    """Pictogram item persistent class"""

    visible = FieldProperty(IPictogramItem['visible'])
    _pictogram_name = FieldProperty(IPictogramItem['pictogram_name'])
    label = FieldProperty(IPictogramItem['label'])
    body = FieldProperty(IPictogramItem['body'])

    @property
    def pictogram_name(self):
        return self._pictogram_name

    @pictogram_name.setter
    def pictogram_name(self, value):
        if value != self._pictogram_name:
            self._pictogram_name = value
            del self.pictogram

    @volatile_property
    def pictogram(self):
        table = query_utility(IPictogramTable)
        if table is not None:
            return table.get(self._pictogram_name)


@adapter_config(context=IPictogramItem, provides=IFormContextPermissionChecker)
def pictogram_item_permission_checker(context):
    """Pictogram item permission checker"""
    content = get_parent(context, IPictogramContainerTarget)
    return IFormContextPermissionChecker(content)


@subscriber(IObjectAddedEvent, context_selector=IPictogramItem)
def handle_added_pictogram_item(event):
    """Handle added pictogram item"""
    content = get_parent(event.object, IPictogramContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@adapter_config(context=IPictogramItem, provides=IContentChecker)
class PictogramItemContentChecker(BaseContentChecker):
    """Pictogram item content checker"""

    @property
    def label(self):
        request = check_request()
        return II18n(self.context).query_attribute('title', request=request)

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language,)
        i18n = II18n(self.context)
        for lang in langs:
            for attr in ('label', 'body'):
                value = i18n.get_attribute(attr, lang, request)
                if not value:
                    field_title = translate(IPictogramItem[attr].title)
                    if len(langs) == 1:
                        output.append(translate(MISSING_VALUE).format(field=field_title))
                    else:
                        output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        field_title = translate(IPictogramItem['pictogram_name'].title)
        if not self.context.pictogram_name:
            output.append(translate(MISSING_VALUE).format(field=field_title))
        else:
            pictogram = self.context.pictogram
            if pictogram is None:
                output.append(translate(ERROR_VALUE).format(field=field_title,
                                                            message=translate(_("Selected pictogram is missing"))))
        return output


#
# Pictogram container classes and adapters
#

@factory_config(IPictogramContainer)
class PictogramContainer(OrderedContainer):
    """Pictogram item container"""

    last_id = 1

    def append(self, value, notify=True):
        key = str(self.last_id)
        if not notify:
            # pre-locate pictogram item to avoid multiple notifications
            locate(value, self, key)
        self[key] = value
        self.last_id += 1
        if not notify:
            # make sure that pictogram item is correctly indexed
            index_object(value)

    def get_visible_items(self):
        yield from filter(lambda x: IPictogramItem(x).visible, self.values())


@adapter_config(context=IPictogramContainerTarget, provides=IPictogramContainer)
def pictogram_container_factory(target):
    """Pictogram container factory"""
    return get_annotation_adapter(target, PICTOGRAM_CONTAINER_KEY, IPictogramContainer,
                                  name='++pictos++')


@adapter_config(name='pictos', context=IPictogramContainerTarget, provides=ITraversable)
class PictogramContainerNamespace(ContextAdapter):
    """Pictogram container ++pictos++ namespace"""

    def traverse(self, name, furtherpath=None):
        return IPictogramContainer(self.context)


@adapter_config(name='pictograms', context=IPictogramContainerTarget, provides=ISublocations)
class PictogramContainerSublocations(ContextAdapter):
    """Pictogram container sub-locations adapter"""

    def sublocations(self):
        return IPictogramContainer(self.context).values()


@adapter_config(name='pictograms', context=IPictogramContainerTarget, provides=IContentChecker)
class PictogramContainerContentChecker(BaseContentChecker):
    """Pictogram container content checker"""

    label = PICTOGRAM_PARAGRAPH_NAME
    sep = '\n'
    weight = 210

    def inner_check(self, request):
        output = []
        registry = request.registry
        for pictogram in IPictogramContainer(self.context).values():
            if not pictogram.visible:
                continue
            for name, checker in sorted(registry.getAdapters((pictogram,), IContentChecker),
                                        key=lambda x: x[1].weight):
                output.append('- {0} :'.format(II18n(pictogram).query_attribute('label', request=request)))
                output.append(checker.get_check_output(request))
        return output


@factory_config(provided=IPictogramParagraph)
@implementer(IBasicIllustrationTarget)
class PictogramParagraph(BaseParagraph):
    """Pictograms paragraph"""

    factory_name = PICTOGRAM_PARAGRAPH_TYPE
    icon_class = 'fa-linode'
    icon_hint = PICTOGRAM_PARAGRAPH_NAME

    renderer = FieldProperty(IPictogramParagraph['renderer'])


@utility_config(name=PICTOGRAM_PARAGRAPH_TYPE, provides=IParagraphFactory)
class PictogramParagraphFactory(BaseParagraphFactory):
    """Pictograms paragraph factory"""

    name = PICTOGRAM_PARAGRAPH_NAME
    content_type = PictogramParagraph


@adapter_config(context=IPictogramParagraph, provides=IContentChecker)
class PictogramParagraphContentChecker(BaseParagraphContentChecker):
    """Pictograms paragraph content checker"""

    @property
    def label(self):
        request = check_request()
        translate = request.localizer.translate
        return II18n(self.context).query_attribute('title', request) or \
            '({0})'.format(translate(self.context.icon_hint).lower())

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language,)
        i18n = II18n(self.context)
        for lang in langs:
            value = i18n.get_attribute('title', lang, request)
            if not value:
                field_title = translate(IPictogramParagraph['title'].title)
                if len(langs) == 1:
                    output.append(translate(MISSING_VALUE).format(field=field_title))
                else:
                    output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        return output


@adapter_config(required=(IPictogramParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONPictogramParagraphConverter(JSONParagraphConverter):
    """JSON pictograms paragraph converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        items = list(IPictogramContainer(self.context).get_visible_items())
        if not items:
            return result
        result['items'] = [
            {
                'label': II18n(item).query_attribute('label', request=self.request),
                'body': II18n(item).query_attribute('body', request=self.request),
                'image': {
                    'src': self.get_image_url(item.pictogram, 'image', params),
                    'content_type': II18n(item.pictogram).query_attribute('image',
                                                                          request=self.request).content_type
                }
            }
            for item in items
            if item.pictogram is not None
        ]
        return result


@vocabulary_config(name=PICTOGRAM_PARAGRAPH_RENDERERS)
class PictogramParagraphRendererVocabulary(RenderersVocabulary):
    """Pictograms paragraph renderers vocabulary"""

    content_interface = IPictogramParagraph
