#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.paragraph import BaseParagraph, BaseParagraphContentChecker, BaseParagraphFactory
from pyams_content.component.paragraph.api import JSONParagraphConverter
from pyams_content.component.paragraph.interfaces import IParagraphFactory
from pyams_content.component.paragraph.interfaces.keypoint import IKeypointsParagraph, KEYPOINTS_PARAGRAPH_NAME, \
    KEYPOINTS_PARAGRAPH_RENDERERS, KEYPOINTS_PARAGRAPH_TYPE
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_LANG_VALUE, MISSING_VALUE
from pyams_content.features.json.interfaces import IJSONConverter
from pyams_content.features.renderer import RenderersVocabulary
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, utility_config
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


@factory_config(provided=IKeypointsParagraph)
class KeypointsParagraph(BaseParagraph):
    """Key points paragraph"""

    factory_name = KEYPOINTS_PARAGRAPH_TYPE
    icon_class = 'fa-list-ol'
    icon_hint = KEYPOINTS_PARAGRAPH_NAME

    body = FieldProperty(IKeypointsParagraph['body'])
    renderer = FieldProperty(IKeypointsParagraph['renderer'])


@utility_config(name=KEYPOINTS_PARAGRAPH_TYPE, provides=IParagraphFactory)
class KeypointsParagraphFactory(BaseParagraphFactory):
    """Key points paragraph factory"""

    name = KEYPOINTS_PARAGRAPH_NAME
    content_type = KeypointsParagraph
    secondary_menu = True


@adapter_config(context=IKeypointsParagraph, provides=IContentChecker)
class KeypointsParagraphContentChecker(BaseParagraphContentChecker):
    """Key points paragraph content checker"""

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language, )
        i18n = II18n(self.context)
        for lang in langs:
            value = i18n.get_attribute('body', lang, request)
            if not value:
                field_title = translate(IKeypointsParagraph['body'].title)
                if len(langs) == 1:
                    output.append(translate(MISSING_VALUE).format(field=field_title))
                else:
                    output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        return output


@vocabulary_config(name=KEYPOINTS_PARAGRAPH_RENDERERS)
class KeypointsParagraphRendererVocabulary(RenderersVocabulary):
    """Key points paragraph renderers vocabulary"""

    content_interface = IKeypointsParagraph


@adapter_config(required=(IKeypointsParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONKeypointsParagraphConverter(JSONParagraphConverter):
    """JSON keypoints paragraph converetr"""

    def convert_content(self, params):
        result = super().convert_content(params)
        body = II18n(self.context).query_attribute('body', request=self.request)
        if body:
            result['items'] = list(map(str.strip, body.split('\n')))
        self.get_i18n_attribute(self.context, 'body', params.get('lang'), result)
        return result
