#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.events import subscriber
from zope.interface import implementer
from zope.lifecycleevent.interfaces import IObjectAddedEvent, IObjectModifiedEvent
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.extfile.interfaces import IExtFileContainerTarget
from pyams_content.component.illustration.interfaces import IIllustrationTarget
from pyams_content.component.links.interfaces import ILinkContainerTarget
from pyams_content.component.paragraph import BaseParagraph, BaseParagraphContentChecker, \
    BaseParagraphFactory
from pyams_content.component.paragraph.api import JSONParagraphConverter
from pyams_content.component.paragraph.association import check_associations
from pyams_content.component.paragraph.interfaces import IParagraphFactory
from pyams_content.component.paragraph.interfaces.html import HTML_PARAGRAPH_NAME, \
    HTML_PARAGRAPH_RENDERERS, HTML_PARAGRAPH_TYPE, IHTMLParagraph, IRawParagraph, \
    RAW_PARAGRAPH_NAME, RAW_PARAGRAPH_RENDERERS, RAW_PARAGRAPH_TYPE
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_LANG_VALUE, \
    MISSING_VALUE
from pyams_content.features.json.interfaces import IJSONConverter
from pyams_content.features.renderer import RenderersVocabulary
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, utility_config
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


__docformat__ = 'restructuredtext'


#
# Raw HTML paragraph
#

@factory_config(provided=IRawParagraph)
class RawParagraph(BaseParagraph):
    """Raw HTML paragraph"""

    factory_name = RAW_PARAGRAPH_TYPE
    icon_class = 'fa-code'
    icon_hint = RAW_PARAGRAPH_NAME

    body = FieldProperty(IRawParagraph['body'])
    renderer = FieldProperty(IRawParagraph['renderer'])


@utility_config(name=RAW_PARAGRAPH_TYPE, provides=IParagraphFactory)
class RawParagraphFactory(BaseParagraphFactory):
    """Raw paragraph factory"""

    name = RAW_PARAGRAPH_NAME
    content_type = RawParagraph


@adapter_config(context=IRawParagraph, provides=IContentChecker)
class RawParagraphContentChecker(BaseParagraphContentChecker):
    """Raw HTML paragraph content checker"""

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language,)
        i18n = II18n(self.context)
        for lang in langs:
            value = i18n.get_attribute('body', lang, request)
            if not value:
                field_title = translate(IRawParagraph['body'].title)
                if len(langs) == 1:
                    output.append(translate(MISSING_VALUE).format(field=field_title))
                else:
                    output.append(
                        translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        return output


@vocabulary_config(name=RAW_PARAGRAPH_RENDERERS)
class RawParagraphRendererVocabulary(RenderersVocabulary):
    """Raw HTML paragraph renderers vocabulary"""

    content_interface = IRawParagraph


@adapter_config(required=(IRawParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONRawParagraphConverter(JSONParagraphConverter):
    """JSON raw paragraph converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        self.get_i18n_attribute(self.context, 'body', params.get('lang'), result)
        return result


#
# HTML paragraph
#

@factory_config(provided=IHTMLParagraph)
@implementer(IIllustrationTarget, IExtFileContainerTarget, ILinkContainerTarget)
class HTMLParagraph(BaseParagraph):
    """HTML paragraph"""

    factory_name = HTML_PARAGRAPH_TYPE
    icon_class = 'fa-font'
    icon_hint = HTML_PARAGRAPH_NAME

    body = FieldProperty(IHTMLParagraph['body'])
    renderer = FieldProperty(IHTMLParagraph['renderer'])


@utility_config(name=HTML_PARAGRAPH_TYPE, provides=IParagraphFactory)
class HTMLParagraphFactory(BaseParagraphFactory):
    """HTML paragraph factory"""

    name = HTML_PARAGRAPH_NAME
    content_type = HTMLParagraph


@subscriber(IObjectAddedEvent, context_selector=IHTMLParagraph)
def handle_added_html_paragraph(event):
    """Check for new associations from added paragraph"""
    paragraph = event.object
    for lang, body in (paragraph.body or {}).items():
        check_associations(paragraph, body, lang, notify=False)


@subscriber(IObjectModifiedEvent, context_selector=IHTMLParagraph)
def handle_modified_html_paragraph(event):
    """Check for new associations from modified paragraph"""
    paragraph = event.object
    for lang, body in (paragraph.body or {}).items():
        check_associations(paragraph, body, lang, notify=False)


@adapter_config(context=IHTMLParagraph, provides=IContentChecker)
class HTMLParagraphContentChecker(BaseParagraphContentChecker):
    """HTML paragraph content checker"""

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language,)
        i18n = II18n(self.context)
        for lang in langs:
            value = i18n.get_attribute('body', lang, request)
            if not value:
                field_title = translate(IHTMLParagraph['body'].title)
                if len(langs) == 1:
                    output.append(translate(MISSING_VALUE).format(field=field_title))
                else:
                    output.append(
                        translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        return output


@vocabulary_config(name=HTML_PARAGRAPH_RENDERERS)
class HTMLParagraphRendererVocabulary(RenderersVocabulary):
    """HTML paragraph renderers vocabulary"""

    content_interface = IHTMLParagraph


@adapter_config(required=(IHTMLParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONHTMLParagraphConverter(JSONParagraphConverter):
    """JSON HTML paragraph converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        self.get_html_attribute(self.context, 'body', params.get('lang'), result)
        return result
