#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.events import subscriber
from zope.interface import implementer
from zope.lifecycleevent import IObjectAddedEvent, IObjectModifiedEvent
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.extfile.interfaces import IExtFileContainerTarget
from pyams_content.component.illustration.interfaces import IIllustrationTarget
from pyams_content.component.links.interfaces import ILinkContainerTarget
from pyams_content.component.paragraph import BaseParagraph, BaseParagraphContentChecker, \
    BaseParagraphFactory
from pyams_content.component.paragraph.api import JSONParagraphConverter
from pyams_content.component.paragraph.association import check_associations
from pyams_content.component.paragraph.interfaces import IParagraphFactory
from pyams_content.component.paragraph.interfaces.frame import FRAME_PARAGRAPH_NAME, \
    FRAME_PARAGRAPH_RENDERERS, FRAME_PARAGRAPH_TYPE, IFrameParagraph
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_LANG_VALUE, \
    MISSING_VALUE
from pyams_content.features.json.interfaces import IJSONConverter
from pyams_content.features.renderer import RenderersVocabulary
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, utility_config
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


#
# Frame paragraph
#

@factory_config(provided=IFrameParagraph)
@implementer(IIllustrationTarget, IExtFileContainerTarget, ILinkContainerTarget)
class FrameParagraph(BaseParagraph):
    """Framed text paragraph"""

    factory_name = FRAME_PARAGRAPH_TYPE
    icon_class = 'fa-window-maximize'
    icon_hint = FRAME_PARAGRAPH_NAME

    body = FieldProperty(IFrameParagraph['body'])
    renderer = FieldProperty(IFrameParagraph['renderer'])


@utility_config(name=FRAME_PARAGRAPH_TYPE, provides=IParagraphFactory)
class FrameParagraphFactory(BaseParagraphFactory):
    """Framed text paragraph factory"""

    name = FRAME_PARAGRAPH_NAME
    content_type = FrameParagraph
    secondary_menu = True


@adapter_config(context=IFrameParagraph, provides=IContentChecker)
class FrameParagraphContentChecker(BaseParagraphContentChecker):
    """Framed text paragraph content checker"""

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language,)
        i18n = II18n(self.context)
        for lang in langs:
            for attr in ('title', 'body'):
                value = i18n.get_attribute(attr, lang, request)
                if not value:
                    field_title = translate(IFrameParagraph[attr].title)
                    if len(langs) == 1:
                        output.append(translate(MISSING_VALUE).format(field=field_title))
                    else:
                        output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        return output


@vocabulary_config(name=FRAME_PARAGRAPH_RENDERERS)
class FrameParagraphRendererVocabulary(RenderersVocabulary):
    """Framed text paragraph renderers vocabulary"""

    content_interface = IFrameParagraph


@subscriber(IObjectAddedEvent, context_selector=IFrameParagraph)
def handle_added_frame_paragraph(event):
    """Check for new associations from added paragraph"""
    paragraph = event.object
    for lang, body in (paragraph.body or {}).items():
        check_associations(paragraph, body, lang, notify=False)


@subscriber(IObjectModifiedEvent, context_selector=IFrameParagraph)
def handle_modified_frame_paragraph(event):
    """Check for new associations from modified paragraph"""
    paragraph = event.object
    for lang, body in (paragraph.body or {}).items():
        check_associations(paragraph, body, lang, notify=False)


@adapter_config(required=(IFrameParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONFrameParagraphConverter(JSONParagraphConverter):
    """JSON frame paragraph converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        self.get_html_attribute(self.context, 'body', params.get('lang'), result)
        return result
