#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.interface import alsoProvides
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.paragraph import BaseParagraph, BaseParagraphContentChecker, BaseParagraphFactory
from pyams_content.component.paragraph.api import JSONParagraphConverter
from pyams_content.component.paragraph.interfaces import IParagraphFactory
from pyams_content.component.paragraph.interfaces.contact import CONTACT_PARAGRAPH_NAME, CONTACT_PARAGRAPH_RENDERERS, \
    CONTACT_PARAGRAPH_TYPE, IContactParagraph, have_gis
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_LANG_VALUE, MISSING_VALUE
from pyams_content.features.json.interfaces import IJSONConverter
from pyams_content.features.renderer import RenderersVocabulary
from pyams_file.interfaces import IImage, IResponsiveImage
from pyams_file.property import FileProperty
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, utility_config
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config

try:
    from pyams_gis.schema import GeoPointField
except ImportError:
    have_gis = False
else:
    from pyams_gis.interfaces import WGS84
    have_gis = True


@factory_config(provided=IContactParagraph)
class ContactParagraph(BaseParagraph):
    """Contact paragraph"""

    factory_name = CONTACT_PARAGRAPH_TYPE
    icon_class = 'fa-id-card-o'
    icon_hint = CONTACT_PARAGRAPH_NAME

    name = FieldProperty(IContactParagraph['name'])
    charge = FieldProperty(IContactParagraph['charge'])
    company = FieldProperty(IContactParagraph['company'])
    contact_email = FieldProperty(IContactParagraph['contact_email'])
    contact_form = FieldProperty(IContactParagraph['contact_form'])
    phone_number = FieldProperty(IContactParagraph['phone_number'])
    _photo = FileProperty(IContactParagraph['photo'])

    if have_gis:
        gps_location = FieldProperty(IContactParagraph['gps_location'])

    address = FieldProperty(IContactParagraph['address'])
    renderer = FieldProperty(IContactParagraph['renderer'])

    @property
    def photo(self):
        return self._photo

    @photo.setter
    def photo(self, value):
        self._photo = value
        if IImage.providedBy(self._photo):
            alsoProvides(self._photo, IResponsiveImage)
            
    @photo.deleter
    def photo(self):
        del self._photo


@utility_config(name=CONTACT_PARAGRAPH_TYPE, provides=IParagraphFactory)
class ContactParagraphFactory(BaseParagraphFactory):
    """Contact paragraph factory"""

    name = CONTACT_PARAGRAPH_NAME
    content_type = ContactParagraph
    secondary_menu = True


@adapter_config(context=IContactParagraph, provides=IContentChecker)
class ContactParagraphContentChecker(BaseParagraphContentChecker):
    """Contact paragraph content checker"""

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language, )
        i18n = II18n(self.context)
        for lang in langs:
            value = i18n.get_attribute('charge', lang, request)
            if not value:
                field_title = translate(IContactParagraph['charge'].title)
                if len(langs) == 1:
                    output.append(translate(MISSING_VALUE).format(field=field_title))
                else:
                    output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        attrs = ('name', 'photo', 'address')
        if have_gis:
            attrs += ('gps_location', )
        for attr in attrs:
            value = getattr(self.context, attr, None)
            if not value:
                field_title = translate(IContactParagraph[attr].title)
                output.append(translate(MISSING_VALUE).format(field=field_title))
        return output


@vocabulary_config(name=CONTACT_PARAGRAPH_RENDERERS)
class ContactParagraphRendererVocabulary(RenderersVocabulary):
    """Contact paragraph renderers vocabulary"""

    content_interface = IContactParagraph


@adapter_config(required=(IContactParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONContactParagraphConverter(JSONParagraphConverter):
    """JSON contact paragraph converter"""

    @property
    def conversion_target(self):
        return None

    def convert_content(self, params):
        result = super().convert_content(params)
        self.get_attribute(self.context, 'name', result)
        self.get_i18n_attribute(self.context, 'charge', params.get('lang'), result)
        self.get_attribute(self.context, 'company', result)
        self.get_attribute(self.context, 'address', result)
        self.get_attribute(self.context, 'contact_email', result)
        self.get_attribute(self.context, 'phone_number', result)
        photo_params = params.copy()
        photo_params['display_name'] = 'portrait'
        photo_params['display_size'] = 'w480'
        self.get_image_url(self.context, 'photo', photo_params)
        if have_gis and self.context.gps_location:
            srid = params.get('srid', WGS84)
            try:
                coords = self.context.gps_location.get_coordinates(srid)
                result['gps_location'] = {
                    'longitude': float(coords[0]),
                    'latitude': float(coords[1]),
                    'srid': srid
                }
            except NotImplementedError:  # unknown SRID
                pass
        return result
