#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.illustration.interfaces import IBasicIllustrationTarget
from pyams_content.component.paragraph import BaseParagraph, BaseParagraphContentChecker, BaseParagraphFactory
from pyams_content.component.paragraph.api import JSONParagraphConverter
from pyams_content.component.paragraph.interfaces import IParagraphFactory
from pyams_content.component.paragraph.interfaces.audio import AUDIO_PARAGRAPH_NAME, AUDIO_PARAGRAPH_RENDERERS, \
    AUDIO_PARAGRAPH_TYPE, IAudioParagraph
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_LANG_VALUE, MISSING_VALUE
from pyams_content.features.json.interfaces import IJSONConverter
from pyams_content.features.renderer import RenderersVocabulary
from pyams_file.property import FileProperty
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, utility_config
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_utils.vocabulary import vocabulary_config


@factory_config(provided=IAudioParagraph)
@implementer(IBasicIllustrationTarget)
class AudioParagraph(BaseParagraph):
    """Audio paragraph class"""

    factory_name = AUDIO_PARAGRAPH_TYPE
    icon_class = 'fa-volume-up'
    icon_hint = AUDIO_PARAGRAPH_NAME

    data = FileProperty(IAudioParagraph['data'])
    author = FieldProperty(IAudioParagraph['author'])
    description = FieldProperty(IAudioParagraph['description'])
    renderer = FieldProperty(IAudioParagraph['renderer'])


@utility_config(name=AUDIO_PARAGRAPH_TYPE,
                provides=IParagraphFactory)
class AudioParagraphFactory(BaseParagraphFactory):
    """Audio paragraph factory"""

    name = AUDIO_PARAGRAPH_NAME
    content_type = AudioParagraph


@adapter_config(required=IAudioParagraph,
                provides=IContentChecker)
class AudioParagraphContentChecker(BaseParagraphContentChecker):
    """Audio paragraph content checker"""

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language, )
        i18n = II18n(self.context)
        for lang in langs:
            value = i18n.get_attribute('title', lang, request)
            if not value:
                field_title = translate(IAudioParagraph['title'].title)
                if len(langs) == 1:
                    output.append(translate(MISSING_VALUE).format(field=field_title))
                else:
                    output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        for attr in ('author', 'data'):
            value = getattr(self.context, attr)
            if not value:
                output.append(translate(MISSING_VALUE).format(field=translate(IAudioParagraph[attr].title)))
        return output


@vocabulary_config(name=AUDIO_PARAGRAPH_RENDERERS)
class AudioParagraphRendererVocabulary(RenderersVocabulary):
    """Audio paragraph renderers vocabulary"""

    content_interface = IAudioParagraph


@adapter_config(required=(IAudioParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONAudioParagraphConverter(JSONParagraphConverter):
    """JSON audio paragraph converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        if self.context.data:
            result['src'] = absolute_url(self.context.data, self.request)
        self.get_attribute(self.context, 'author', result)
        self.get_i18n_attribute(self.context, 'description', params.get('lang'), result)
        return result
